<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/** خريطة أدوار الموظفين (كود → تسمية عربية) */
$ROLE_LABELS = [
  'manager'    => 'مدير فرع',
  'supervisor' => 'مشرف شِفت',
  'sales'      => 'بائع/خدمة عملاء',
  'cashier'    => 'كاشير',
  'inventory'  => 'أمين مخزن',
  'buyer'      => 'مشتريات',
];
$ALLOWED_ROLES = array_keys($ROLE_LABELS);

/* ========== إضافة أو تعديل موظف ========== */
if ($_SERVER["REQUEST_METHOD"] === "POST") {
    $id       = $_POST['id'] ?? null;
    $name     = trim($_POST['EmpName'] ?? '');
    $phone    = trim($_POST['Phone'] ?? '');
    $email    = trim($_POST['Email'] ?? '');
    // ندعم الاسم القديم Role لو النموذج القديم لسه موجود
    $role     = $_POST['employee_role'] ?? $_POST['Role'] ?? 'sales';
    $role     = in_array($role, $ALLOWED_ROLES, true) ? $role : 'sales';

    $branch   = $_POST['BranchID'] !== '' ? (int)$_POST['BranchID'] : null;
    $isActive = isset($_POST['IsActive']) ? 1 : 0;
    $hireDate = $_POST['HireDate'] !== '' ? $_POST['HireDate'] : null;
    $notes    = trim($_POST['Notes'] ?? '');

    if ($name === "") {
        $error = "❌ برجاء إدخال اسم الموظف";
    } else {
        try {
            if ($id) {
                // تحديث
                $stmt = $pdo->prepare("
                    UPDATE Employees
                    SET EmpName=?, Phone=?, Email=?, employee_role=?, BranchID=?, IsActive=?, HireDate=?, Notes=?
                    WHERE EmpID=?
                ");
                $stmt->execute([$name, $phone, $email, $role, $branch, $isActive, $hireDate, $notes, $id]);
                $success = "✅ تم تعديل بيانات الموظف";
            } else {
                // إضافة
                $stmt = $pdo->prepare("
                    INSERT INTO Employees (EmpName, Phone, Email, employee_role, BranchID, IsActive, HireDate, Notes)
                    VALUES (?,?,?,?,?,?,?,?)
                ");
                $stmt->execute([$name, $phone, $email, $role, $branch, $isActive, $hireDate, $notes]);
                $success = "✅ تم إضافة الموظف";
            }
        } catch (PDOException $e) {
            $error = "⚠️ خطأ: " . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
        }
    }
}

/* ========== حذف موظف ========== */
if (isset($_GET['delete'])) {
    $id = (int) $_GET['delete'];
    $pdo->prepare("DELETE FROM Employees WHERE EmpID=?")->execute([$id]);
    $success = "🗑️ تم حذف الموظف";
}

/* ========== جلب الفروع ========== */
$branches = $pdo->query("SELECT BranchID, BranchName FROM Branches ORDER BY BranchName")
                ->fetchAll(PDO::FETCH_KEY_PAIR);

/* ========== جلب الموظفين ========== */
$employees = $pdo->query("
    SELECT e.*, b.BranchName
    FROM Employees e
    LEFT JOIN Branches b ON e.BranchID=b.BranchID
    ORDER BY e.EmpID DESC
")->fetchAll();

/* ========== الكروت (إحصائيات) ========== */
$totalEmployees = (int)$pdo->query("SELECT COUNT(*) FROM Employees")->fetchColumn();
$totalSales     = (int)$pdo->query("SELECT COUNT(*) FROM Employees WHERE employee_role='sales'")->fetchColumn();
$totalCashiers  = (int)$pdo->query("SELECT COUNT(*) FROM Employees WHERE employee_role='cashier'")->fetchColumn();
$totalInventory = (int)$pdo->query("SELECT COUNT(*) FROM Employees WHERE employee_role='inventory'")->fetchColumn();
$lastEmployee   = $pdo->query("SELECT EmpName FROM Employees ORDER BY EmpID DESC LIMIT 1")->fetchColumn();
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>إدارة الموظفين</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    body { font-family:"Tajawal",sans-serif; background:#f8f9fa; }
    .card { border-radius:12px; box-shadow:0 4px 10px rgba(0,0,0,0.1); }
    .card-header { background:linear-gradient(135deg,#0f2027,#203a43,#2c5364); color:#fff; font-weight:bold; }
    .btn-custom { border-radius:25px; }
  </style>
</head>
<body>
<div class="container py-4">

<header class="mb-4">
  <div class="d-flex justify-content-between align-items-center p-3 bg-dark text-white shadow-sm">
    <!-- يمين: بيانات المستخدم -->
    <div class="d-flex align-items-center gap-2">
      <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>"
           alt="Avatar" class="rounded-circle border border-2 border-light"
           style="width:40px;height:40px;object-fit:cover;">
      <span class="fw-bold"><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Admin') ?></span>
      <a href="logout.php" class="btn btn-sm btn-outline-light">خروج</a>
    </div>

    <!-- وسط: عنوان الصفحة -->
    <h1 class="h5 m-0">الموظفون</h1>

    <!-- شمال: زر الوضع الليلي + رجوع -->
    <div class="d-flex align-items-center gap-2">
      <a href="dashboard.php" class="btn btn-sm btn-secondary">
        <i class="fa-solid fa-house"></i> الرئيسية
      </a>
      <button id="darkSwitch" class="btn btn-sm btn-success">🌙</button>
    </div>
  </div>
</header>

<!-- الكروت -->
<div class="row text-center mb-4">
  <div class="col-md-2 mb-3">
    <div class="card text-white bg-primary shadow">
      <div class="card-body">
        <h6>👥 عدد الموظفين</h6>
        <h2><?= $totalEmployees ?></h2>
      </div>
    </div>
  </div>
  <div class="col-md-2 mb-3">
    <div class="card text-white bg-success shadow">
      <div class="card-body">
        <h6>🛍️ البائعون</h6>
        <h2><?= $totalSales ?></h2>
      </div>
    </div>
  </div>
  <div class="col-md-2 mb-3">
    <div class="card text-white bg-info shadow">
      <div class="card-body">
        <h6>💳 الكاشير</h6>
        <h2><?= $totalCashiers ?></h2>
      </div>
    </div>
  </div>
  <div class="col-md-2 mb-3">
    <div class="card text-white bg-warning shadow">
      <div class="card-body">
        <h6>📦 المخزون</h6>
        <h2><?= $totalInventory ?></h2>
      </div>
    </div>
  </div>
  <div class="col-md-4 mb-3">
    <div class="card text-white bg-dark shadow">
      <div class="card-body">
        <h6>🆕 آخر موظف</h6>
        <h4><?= htmlspecialchars($lastEmployee ?: '—') ?></h4>
      </div>
    </div>
  </div>
</div>

<div class="card">
  <div class="card-header">👨‍💼 إدارة الموظفين</div>
  <div class="card-body">

    <?php if ($success): ?>
      <div class="alert alert-success text-center"><?= $success ?></div>
    <?php elseif ($error): ?>
      <div class="alert alert-danger text-center"><?= $error ?></div>
    <?php endif; ?>

    <!-- نموذج إضافة / تعديل -->
    <form method="post" class="row g-3 mb-4">
      <input type="hidden" name="id" id="emp_id">

      <div class="col-md-4">
        <input type="text" name="EmpName" id="emp_name" class="form-control" placeholder="اسم الموظف" required>
      </div>
      <div class="col-md-4">
        <input type="text" name="Phone" id="emp_phone" class="form-control" placeholder="الهاتف">
      </div>
      <div class="col-md-4">
        <input type="email" name="Email" id="emp_email" class="form-control" placeholder="البريد الإلكتروني">
      </div>

      <div class="col-md-3">
        <select name="employee_role" id="emp_role" class="form-select" required>
          <option value="sales">بائع/خدمة عملاء</option>
          <option value="cashier">كاشير</option>
          <option value="inventory">أمين مخزن</option>
          <option value="manager">مدير فرع</option>
          <option value="supervisor">مشرف شِفت</option>
          <option value="buyer">مشتريات</option>
        </select>
      </div>

      <div class="col-md-3">
        <select name="BranchID" id="emp_branch" class="form-select">
          <option value="">اختر الفرع</option>
          <?php foreach ($branches as $id=>$name): ?>
            <option value="<?= (int)$id ?>"><?= htmlspecialchars($name, ENT_QUOTES, 'UTF-8') ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-md-2 d-flex align-items-center">
        <input type="checkbox" name="IsActive" id="emp_active" class="form-check-input me-2" checked>
        <label for="emp_active" class="form-check-label">نشط</label>
      </div>

      <div class="col-md-4">
        <input type="date" name="HireDate" id="emp_hiredate" class="form-control">
      </div>

      <div class="col-md-12">
        <input type="text" name="Notes" id="emp_notes" class="form-control" placeholder="ملاحظات">
      </div>

      <div class="col-md-12 text-end">
        <button type="submit" class="btn btn-primary btn-custom px-4">💾 حفظ</button>
      </div>
    </form>

    <!-- جدول عرض الموظفين -->
    <div class="table-responsive">
      <table class="table table-bordered text-center align-middle">
        <thead class="table-dark">
          <tr>
            <th>#</th>
            <th>الاسم</th>
            <th>الهاتف</th>
            <th>البريد</th>
            <th>الوظيفة</th>
            <th>الفرع</th>
            <th>الحالة</th>
            <th>تاريخ التعيين</th>
            <th>ملاحظات</th>
            <th>إجراءات</th>
          </tr>
        </thead>
        <tbody>
        <?php if (!$employees): ?>
          <tr><td colspan="10" class="text-muted">لا توجد بيانات.</td></tr>
        <?php else: foreach ($employees as $e): ?>
          <?php
            $roleCode = $e['employee_role'] ?? 'sales';
            $roleName = $ROLE_LABELS[$roleCode] ?? $roleCode;
          ?>
          <tr data-row='<?= htmlspecialchars(json_encode($e), ENT_QUOTES, "UTF-8") ?>'>
            <td><?= (int)$e['EmpID'] ?></td>
            <td><?= htmlspecialchars($e['EmpName'], ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($e['Phone'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($e['Email'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($roleName, ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($e['BranchName'] ?? '—', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= ((int)$e['IsActive'] === 1) ? '<span class="badge bg-success">نشط</span>' : '<span class="badge bg-danger">موقوف</span>' ?></td>
            <td><?= htmlspecialchars($e['HireDate'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($e['Notes'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
            <td>
              <button class="btn btn-sm btn-warning" onclick="openEdit(this)">✏️</button>
              <a href="?delete=<?= (int)$e['EmpID'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('هل أنت متأكد من الحذف؟')">🗑️</a>
            </td>
          </tr>
        <?php endforeach; endif; ?>
        </tbody>
      </table>
    </div>

  </div>
</div>

</div> <!-- /container -->

<script>
  // Dark mode toggle
  const body = document.body;
  const darkSwitch = document.getElementById("darkSwitch");
  if (localStorage.getItem("darkMode") === "true") { body.classList.add("dark"); darkSwitch.textContent = "☀️"; }
  darkSwitch?.addEventListener("click", ()=>{
    body.classList.toggle("dark");
    const isDark = body.classList.contains("dark");
    localStorage.setItem("darkMode", isDark);
    darkSwitch.textContent = isDark ? "☀️" : "🌙";
  });

  function openEdit(btn){
    const tr = btn.closest('tr');
    const data = JSON.parse(tr.getAttribute('data-row'));
    document.getElementById("emp_id").value       = data.EmpID ?? '';
    document.getElementById("emp_name").value     = data.EmpName ?? '';
    document.getElementById("emp_phone").value    = data.Phone ?? '';
    document.getElementById("emp_email").value    = data.Email ?? '';
    document.getElementById("emp_role").value     = data.employee_role ?? 'sales';
    document.getElementById("emp_branch").value   = data.BranchID ?? '';
    document.getElementById("emp_active").checked = (parseInt(data.IsActive ?? 1) === 1);
    document.getElementById("emp_hiredate").value = data.HireDate ?? '';
    document.getElementById("emp_notes").value    = data.Notes ?? '';
    window.scrollTo({ top: 0, behavior: 'smooth' });
  }
</script>
</body>
</html>
