<?php
// inventory.php — Luxury Inventory Page (RTL)
// -------------------------------------------
// ملاحظات سريعة:
// 1) حدّث إعدادات الاتصال بقاعدة البيانات تحت (أو اتركها فارغة للتجربة ببيانات وهمية).
// 2) الجدول المقترح: products (id, name, brand, category, color, size, material, gender, price, old_price, rating, in_stock, on_sale, image_url, popularity, created_at)
// 3) الصفحة تدعم GET للفلترة و sort و pagination + تفضيلات تُحفظ في الجلسة/كوكيز.
// -------------------------------------------

session_start();

// ====== إعدادات قاعدة البيانات (حدّثها عندك) ======
$DB_DSN  = getenv('DB_DSN') ?: 'mysql:host=localhost;dbname=fashion;charset=utf8mb4';
$DB_USER = getenv('DB_USER') ?: 'root';
$DB_PASS = getenv('DB_PASS') ?: '';
$USE_DB  = false; // غيّرها إلى true بعد ضبط الاتصال فعلياً

$pdo = null;
if ($USE_DB) {
    try {
        $pdo = new PDO($DB_DSN, $DB_USER, $DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]);
    } catch (Exception $e) {
        $pdo = null;
    }
}

// ====== أدوات مساعدة ======
function get_param($key, $default=null, $allowed=null) {
    if (!isset($_GET[$key])) return $default;
    $v = $_GET[$key];
    if (is_array($v)) {
        $v = array_values(array_filter($v, fn($x)=>$x!=='' && $x!==null));
        if ($allowed) $v = array_values(array_intersect($v, $allowed));
        return $v ?: $default;
    } else {
        $v = trim($v);
        if ($allowed && !in_array($v, $allowed)) return $default;
        return $v === '' ? $default : $v;
    }
}
function int_param($key, $default, $min=1, $max=200) {
    $v = isset($_GET[$key]) ? intval($_GET[$key]) : $default;
    return max($min, min($max, $v));
}
function esc($s){ return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

// ====== قراءة بارامترات الفلترة/الترتيب ======
$categories = ['فساتين','قمصان','تيشيرت','بناطيل','جاكيت','أحذية','حقائب','إكسسوارات'];
$brands     = ['Aurora','Valente','Noir&Co','Lumé','Mare','Haven','Seren','Arcadia'];
$colors     = ['أسود','أبيض','ذهبي','فضي','أحمر','أزرق','أخضر','بيج','وردي'];
$sizes      = ['XS','S','M','L','XL','XXL'];
$materials  = ['قطن','كتان','صوف','حرير','جلد','بوليستر'];
$genders    = ['سيدات','رجال','أطفال','يونيسكس'];

$q          = get_param('q', '');
$f_cat      = get_param('category', [], $categories);
$f_brand    = get_param('brand', [], $brands);
$f_color    = get_param('color', [], $colors);
$f_size     = get_param('size', [], $sizes);
$f_material = get_param('material', [], $materials);
$f_gender   = get_param('gender', [], $genders);
$price_min  = isset($_GET['price_min']) && $_GET['price_min']!=='' ? max(0, floatval($_GET['price_min'])) : null;
$price_max  = isset($_GET['price_max']) && $_GET['price_max']!=='' ? max(0, floatval($_GET['price_max'])) : null;
$rating_min = isset($_GET['rating_min']) ? max(0, min(5, floatval($_GET['rating_min']))) : null;
$in_stock   = isset($_GET['in_stock']) ? 1 : 0;
$on_sale    = isset($_GET['on_sale']) ? 1 : 0;

$sort = get_param('sort', 'newest', ['newest','price_asc','price_desc','popular','rating_desc']);
$view = get_param('view', 'grid', ['grid','list']);
$per  = int_param('per', 24, 8, 60);
$page = int_param('page', 1, 1, 1000);

// ====== تفضيلات المستخدم (جلسة/كوكيز) ======
if (!isset($_SESSION['prefs'])) {
    // جرّب من الكوكيز
    if (!empty($_COOKIE['lux_prefs'])) {
        $decoded = json_decode($_COOKIE['lux_prefs'], true);
        if (is_array($decoded)) $_SESSION['prefs'] = $decoded;
    }
}
$_SESSION['prefs'] = $_SESSION['prefs'] ?? [
    'sizes' => ['M','L'],
    'colors'=> ['أسود','بيج'],
    'brands'=> ['Noir&Co','Lumé'],
    'gender'=> 'سيدات'
];

// AJAX: حفظ تفضيلات
if (($_SERVER['REQUEST_METHOD'] === 'POST') && (isset($_GET['action']) && $_GET['action']==='save_prefs')) {
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $prefs = [
        'sizes'  => array_values(array_intersect($input['sizes']  ?? [], $sizes)),
        'colors' => array_values(array_intersect($input['colors'] ?? [], $colors)),
        'brands' => array_values(array_intersect($input['brands'] ?? [], $brands)),
        'gender' => in_array($input['gender'] ?? '', $genders) ? $input['gender'] : null
    ];
    $_SESSION['prefs'] = array_filter($prefs, fn($v)=>$v!==null);
    setcookie('lux_prefs', json_encode($_SESSION['prefs'], JSON_UNESCAPED_UNICODE), time()+60*60*24*30, '/');
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['ok'=>true,'prefs'=>$_SESSION['prefs']], JSON_UNESCAPED_UNICODE);
    exit;
}

// ====== بيانات افتراضية إذا لا يوجد DB ======
function seed_products(): array {
    $mock = [];
    $sampleImgs = [
        'https://images.unsplash.com/photo-1520975922284-5f18f05f5c65',
        'https://images.unsplash.com/photo-1520975693416-35a3d2c5a8e3',
        'https://images.unsplash.com/photo-1518546305927-5a555bb7020d',
        'https://images.unsplash.com/photo-1540575467063-178a50c2df87',
        'https://images.unsplash.com/photo-1542060748-10c28b62716b',
        'https://images.unsplash.com/photo-1523380744952-b7bda12b76f8',
        'https://images.unsplash.com/photo-1539533113208-f6dfc1f3c183',
        'https://images.unsplash.com/photo-1520975914082-04e5a2f2ef58',
    ];
    $id=1;
    foreach (['Aurora','Valente','Noir&Co','Lumé'] as $br) {
        foreach (['فساتين','تيشيرت','قمصان','بناطيل'] as $cat) {
            $price = rand(40,220) * 1.0;
            $old   = (rand(0,1) ? $price + rand(10,60) : null);
            $mock[] = [
                'id'=>$id++,
                'name'=>"$cat $br Signature",
                'brand'=>$br,
                'category'=>$cat,
                'color'=>['أسود','بيج','أبيض','أزرق','ذهبي'][array_rand([0,1,2,3,4])],
                'size'=>$GLOBALS['sizes'][array_rand($GLOBALS['sizes'])],
                'material'=>$GLOBALS['materials'][array_rand($GLOBALS['materials'])],
                'gender'=>$GLOBALS['genders'][array_rand($GLOBALS['genders'])],
                'price'=>$price,
                'old_price'=>$old,
                'rating'=> rand(30,50)/10,
                'in_stock'=> rand(0,100)>10 ? 1:0,
                'on_sale'=> $old ? 1:0,
                'image_url'=> $sampleImgs[array_rand($sampleImgs)],
                'popularity'=> rand(1,1000),
                'created_at'=> time()-rand(0,60*60*24*180),
            ];
        }
    }
    return $mock;
}

// ====== جلب البيانات (DB أو افتراضي) ======
$total = 0;
$items = [];
$all   = [];

if ($pdo) {
    // --- بناء الاستعلام بأمان ---
    $where   = [];
    $params  = [];

    if ($q) { $where[] = "(name LIKE :q OR brand LIKE :q OR category LIKE :q)"; $params[':q'] = "%$q%"; }
    if ($f_cat) { $where[] = "category IN (". implode(',', array_fill(0, count($f_cat), '?')) .")"; $params = array_merge($params, $f_cat); }
    if ($f_brand){ $where[] = "brand IN (". implode(',', array_fill(0, count($f_brand), '?')) .")"; $params = array_merge($params, $f_brand); }
    if ($f_color){ $where[] = "color IN (". implode(',', array_fill(0, count($f_color), '?')) .")"; $params = array_merge($params, $f_color); }
    if ($f_size){ $where[] = "size IN (". implode(',', array_fill(0, count($f_size), '?')) .")"; $params = array_merge($params, $f_size); }
    if ($f_material){ $where[] = "material IN (". implode(',', array_fill(0, count($f_material), '?')) .")"; $params = array_merge($params, $f_material); }
    if ($f_gender){ $where[] = "gender IN (". implode(',', array_fill(0, count($f_gender), '?')) .")"; $params = array_merge($params, $f_gender); }
    if ($price_min!==null){ $where[] = "price >= ?"; $params[] = $price_min; }
    if ($price_max!==null){ $where[] = "price <= ?"; $params[] = $price_max; }
    if ($rating_min!==null){ $where[] = "rating >= ?"; $params[] = $rating_min; }
    if ($in_stock){ $where[] = "in_stock = 1"; }
    if ($on_sale){ $where[] = "on_sale = 1"; }

    $orderBy = match($sort) {
        'price_asc'  => 'price ASC',
        'price_desc' => 'price DESC',
        'popular'    => 'popularity DESC',
        'rating_desc'=> 'rating DESC',
        default      => 'created_at DESC'
    };

    $whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';
    // إجمالي
    $stmt = $pdo->prepare("SELECT COUNT(*) as c FROM products $whereSql");
    $stmt->execute(array_values($params));
    $total = (int)$stmt->fetchColumn();

    $offset = ($page-1)*$per;
    $stmt = $pdo->prepare("SELECT * FROM products $whereSql ORDER BY $orderBy LIMIT ? OFFSET ?");
    $runParams = array_values($params);
    $runParams[] = $per;
    $runParams[] = $offset;
    $stmt->execute($runParams);
    $items = $stmt->fetchAll();

    // مقترحات
    $pref = $_SESSION['prefs'];
    $recWhere = [];
    $recParams = [];
    if (!empty($pref['brands'])) { $recWhere[]="brand IN (".implode(',', array_fill(0, count($pref['brands']), '?')).")"; $recParams = array_merge($recParams, $pref['brands']); }
    if (!empty($pref['colors'])) { $recWhere[]="color IN (".implode(',', array_fill(0, count($pref['colors']), '?')).")"; $recParams = array_merge($recParams, $pref['colors']); }
    if (!empty($pref['sizes']))  { $recWhere[]="size IN (".implode(',', array_fill(0, count($pref['sizes']), '?')).")"; $recParams = array_merge($recParams, $pref['sizes']); }
    if (!empty($pref['gender'])) { $recWhere[]="gender = ?"; $recParams[] = $pref['gender']; }
    $recSql = $recWhere ? ('WHERE '.implode(' OR ', $recWhere)) : '';
    $stmt2 = $pdo->prepare("SELECT * FROM products $recSql ORDER BY popularity DESC, rating DESC LIMIT 12");
    $stmt2->execute(array_values($recParams));
    $recs = $stmt2->fetchAll();
} else {
    // افتراضي
    $all = seed_products();

    // فلترة برمجياً
    $filtered = array_filter($all, function($p) use($q,$f_cat,$f_brand,$f_color,$f_size,$f_material,$f_gender,$price_min,$price_max,$rating_min,$in_stock,$on_sale){
        if ($q && stripos($p['name'].' '.$p['brand'].' '.$p['category'], $q)===false) return false;
        if ($f_cat && !in_array($p['category'], $f_cat)) return false;
        if ($f_brand && !in_array($p['brand'], $f_brand)) return false;
        if ($f_color && !in_array($p['color'], $f_color)) return false;
        if ($f_size && !in_array($p['size'], $f_size)) return false;
        if ($f_material && !in_array($p['material'], $f_material)) return false;
        if ($f_gender && !in_array($p['gender'], $f_gender)) return false;
        if ($price_min!==null && $p['price'] < $price_min) return false;
        if ($price_max!==null && $p['price'] > $price_max) return false;
        if ($rating_min!==null && $p['rating'] < $rating_min) return false;
        if ($in_stock && !$p['in_stock']) return false;
        if ($on_sale && !$p['on_sale']) return false;
        return true;
    });

    // ترتيب
    usort($filtered, function($a,$b) use($sort){
        return match($sort) {
            'price_asc'   => $a['price'] <=> $b['price'],
            'price_desc'  => $b['price'] <=> $a['price'],
            'popular'     => $b['popularity'] <=> $a['popularity'],
            'rating_desc' => $b['rating'] <=> $a['rating'],
            default       => $b['created_at'] <=> $a['created_at'],
        };
    });

    $total = count($filtered);
    $offset = ($page-1)*$per;
    $items = array_slice($filtered, $offset, $per);

    // مقترحات من التفضيلات
    $pref = $_SESSION['prefs'];
    $recs = array_values(array_filter($all, function($p) use($pref){
        $score = 0;
        if (!empty($pref['brands']) && in_array($p['brand'], $pref['brands'])) $score++;
        if (!empty($pref['colors']) && in_array($p['color'], $pref['colors'])) $score++;
        if (!empty($pref['sizes'])  && in_array($p['size'],  $pref['sizes']))  $score++;
        if (!empty($pref['gender']) && $p['gender']===$pref['gender']) $score++;
        return $score >= 2;
    }));
    if (count($recs)<8) { // كمل بالترند
        usort($all, fn($a,$b)=>($b['popularity']<=>$a['popularity']) ?: ($b['rating']<=>$a['rating']));
        $recs = array_slice(array_unique(array_merge($recs,$all), SORT_REGULAR), 0, 12);
    } else {
        usort($recs, fn($a,$b)=>($b['popularity']<=>$a['popularity']) ?: ($b['rating']<=>$a['rating']));
        $recs = array_slice($recs,0,12);
    }
}

// ====== حساب الصفحات ======
$total_pages = max(1, (int)ceil($total / $per));
$base_query = $_GET;
unset($base_query['page']);
function page_url($p){
    $q = $GLOBALS['base_query']; $q['page']=$p; return '?'.http_build_query($q);
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>المخزون | Luxury Fashion</title>
<link rel="preconnect" href="https://fonts.googleapis.com">
<link href="https://fonts.googleapis.com/css2?family=Cairo:wght@300;400;600;700;800&display=swap" rel="stylesheet">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
<style>
:root{
  --bg: #0B0B0C;
  --panel: rgba(255,255,255,0.04);
  --panel-2: rgba(255,255,255,0.06);
  --stroke: rgba(255,255,255,0.12);
  --text: #EEE;
  --muted:#9e9e9e;
  --gold:#C7A254;
  --gold-2:#E3C987;
  --danger:#ff6b6b;
  --success:#66d19e;
  --shadow: 0 10px 30px rgba(0,0,0,.35), inset 0 1px 0 rgba(255,255,255,.05);
}
*{box-sizing:border-box}
html,body{background:linear-gradient(180deg,#0B0B0C 0%,#0E1014 100%); color:var(--text); font-family:'Cairo',system-ui,Segoe UI,Arial; margin:0}
a{color:inherit; text-decoration:none}
.container{max-width:1300px; margin:0 auto; padding:24px}
.topbar{
  display:flex; gap:16px; align-items:center; justify-content:space-between;
  background:linear-gradient(135deg, rgba(255,255,255,.06), rgba(255,255,255,.02));
  border:1px solid var(--stroke); border-radius:18px; padding:14px 18px; box-shadow:var(--shadow); backdrop-filter: blur(8px);
}
.logo{display:flex; align-items:center; gap:12px; font-weight:800; letter-spacing:.5px}
.logo .mark{
  width:34px; height:34px; border-radius:10px; background: radial-gradient(circle at 30% 30%, var(--gold-2), var(--gold) 60%, #8a6c2d 100%);
  box-shadow: 0 6px 16px rgba(199,162,84,.35), inset 0 1px 1px rgba(255,255,255,.3);
}
.search{flex:1; display:flex; gap:10px}
.search input{
  flex:1; background:var(--panel); color:var(--text); border:1px solid var(--stroke); border-radius:12px; padding:12px 14px; outline:none;
}
.search button{
  background:linear-gradient(135deg,var(--gold-2),var(--gold)); color:#1a1407; font-weight:700; border:none; border-radius:12px; padding:12px 16px; cursor:pointer;
  box-shadow: 0 6px 18px rgba(199,162,84,.35);
}
.actions{display:flex; gap:10px; align-items:center}
.actions .pill{
  background:var(--panel); border:1px solid var(--stroke); border-radius:999px; padding:10px 14px; display:flex; align-items:center; gap:8px; cursor:pointer
}
.layout{
  display:grid; grid-template-columns: 300px 1fr; gap:18px; margin-top:18px;
}
@media (max-width: 980px){ .layout{grid-template-columns: 1fr} .sidebar{order:2} .main{order:1} }
.sidebar{
  background:var(--panel-2); border:1px solid var(--stroke); border-radius:18px; padding:18px; box-shadow:var(--shadow);
  position:sticky; top:16px; height:max-content;
}
.sidebar h3{margin:0 0 10px 0; font-size:18px}
.filter-group{border-top:1px dashed var(--stroke); padding-top:12px; margin-top:12px}
.chips{display:flex; flex-wrap:wrap; gap:8px}
.chips label{
  border:1px solid var(--stroke); background:var(--panel); padding:8px 10px; border-radius:10px; cursor:pointer; font-size:13px;
}
.chips input{display:none}
.chips input:checked + span{ background:linear-gradient(135deg, rgba(199,162,84,.25), rgba(199,162,84,.05)); border-color:var(--gold); box-shadow: inset 0 0 0 1px rgba(199,162,84,.35); }
.range{display:flex; gap:8px}
.range input{flex:1; background:var(--panel); border:1px solid var(--stroke); padding:10px 12px; border-radius:10px; color:var(--text)}
.sidebar .submit{
  width:100%; margin-top:14px; padding:12px; border-radius:12px; border:none; cursor:pointer; font-weight:800;
  background:linear-gradient(135deg,var(--gold-2),var(--gold)); color:#1a1407; box-shadow:0 8px 20px rgba(199,162,84,.35)
}
.sidebar .reset{margin-top:8px; width:100%; padding:10px; border-radius:10px; background:var(--panel); border:1px solid var(--stroke); color:var(--text); cursor:pointer}

.toolbar{
  display:flex; gap:10px; flex-wrap:wrap; align-items:center; justify-content:space-between;
  background:var(--panel-2); border:1px solid var(--stroke); border-radius:16px; padding:10px 12px; box-shadow:var(--shadow)
}
.toolbar .left, .toolbar .right{display:flex; gap:10px; align-items:center; flex-wrap:wrap}
.select, .input{
  background:var(--panel); border:1px solid var(--stroke); border-radius:10px; padding:8px 10px; color:var(--text)
}
.view-toggle button{
  background:var(--panel); border:1px solid var(--stroke); padding:8px 10px; border-radius:10px; cursor:pointer
}
.view-toggle .active{ border-color:var(--gold); box-shadow: inset 0 0 0 1px rgba(199,162,84,.35); }

.grid{
  display:grid; gap:16px; margin-top:16px;
}
.grid.grid-view{grid-template-columns: repeat(4,1fr)}
@media (max-width: 1200px){ .grid.grid-view{grid-template-columns: repeat(3,1fr)} }
@media (max-width: 780px){ .grid.grid-view{grid-template-columns: repeat(2,1fr)} }
@media (max-width: 480px){ .grid.grid-view{grid-template-columns: 1fr} }
.grid.list-view{grid-template-columns: 1fr}

.card{
  background:linear-gradient(180deg, rgba(255,255,255,.04), rgba(255,255,255,.02));
  border:1px solid var(--stroke); border-radius:18px; overflow:hidden; position:relative; box-shadow:var(--shadow);
  display:grid; grid-template-columns: 1fr; transition: transform .2s ease, box-shadow .2s ease;
}
.card:hover{ transform: translateY(-4px); box-shadow: 0 20px 40px rgba(0,0,0,.45); }
.card .img{aspect-ratio: 4/5; background:#111 center/cover no-repeat}
.card .badge{ position:absolute; top:10px; left:10px; background:linear-gradient(135deg,var(--gold-2),var(--gold)); color:#1a1407; padding:6px 10px; border-radius:999px; font-weight:800; font-size:12px; box-shadow: 0 6px 16px rgba(199,162,84,.4) }
.card .fav{ position:absolute; top:10px; right:10px; background:var(--panel); border:1px solid var(--stroke); width:36px; height:36px; display:grid; place-items:center; border-radius:50%; cursor:pointer }
.card .body{ padding:12px 14px; display:grid; gap:8px }
.card .name{ font-weight:800; line-height:1.3 }
.meta{display:flex; gap:8px; align-items:center; color:var(--muted); font-size:13px}
.price{display:flex; gap:8px; align-items:baseline}
.price .now{font-weight:900; letter-spacing:.2px}
.price .old{color:var(--muted); text-decoration:line-through}
.actions-row{display:flex; gap:10px}
.btn{
  flex:1; text-align:center; padding:10px 12px; border-radius:10px; cursor:pointer; border:1px solid var(--stroke); background:var(--panel); font-weight:700
}
.btn.primary{ background:linear-gradient(135deg,var(--gold-2),var(--gold)); color:#1a1407; border:none }
.rating{display:inline-flex; align-items:center; gap:6px}
.rating i{ color: var(--gold); }
.list-view .card{ grid-template-columns: 220px 1fr; }
.list-view .card .img{ aspect-ratio: auto; height:100% }

.pager{display:flex; gap:8px; justify-content:center; margin:22px 0}
.pager a, .pager span{
  min-width:38px; text-align:center; padding:8px 10px; border-radius:10px; border:1px solid var(--stroke); background:var(--panel); display:inline-block
}
.pager .active{ border-color:var(--gold); box-shadow: inset 0 0 0 1px rgba(199,162,84,.35); }

.section{margin-top:28px}
.section h2{margin:0 0 10px 0; font-size:20px}
.recs{ display:grid; grid-template-columns: repeat(6,1fr); gap:14px }
@media (max-width: 1200px){ .recs{grid-template-columns: repeat(4,1fr)} }
@media (max-width: 780px){ .recs{grid-template-columns: repeat(2,1fr)} }

.drawer{
  position:fixed; inset:0; display:none; place-items:center; background:rgba(0,0,0,.45); z-index:50;
}
.drawer.active{ display:grid }
.drawer .panel{
  width:min(680px, 95vw); background:linear-gradient(180deg, rgba(255,255,255,.06), rgba(255,255,255,.02));
  border:1px solid var(--stroke); border-radius:18px; padding:18px; box-shadow:var(--shadow)
}
.panel .row{ display:grid; grid-template-columns: 1fr 1fr; gap:12px }
@media (max-width: 620px){ .panel .row{ grid-template-columns: 1fr } }
.panel h3{margin:0 0 10px 0}
.panel .save{ margin-top:12px; width:100%; padding:12px; border-radius:12px; border:none; cursor:pointer; font-weight:900;
  background:linear-gradient(135deg,var(--gold-2),var(--gold)); color:#1a1407; box-shadow:0 10px 24px rgba(199,162,84,.35);
}
.badge-soft{ padding:6px 10px; border-radius:999px; background:rgba(199,162,84,.12); border:1px solid rgba(199,162,84,.35); color:var(--gold); font-size:12px }
.small{font-size:12px; color:var(--muted)}
.hr{height:1px; background:var(--stroke); margin:8px 0}
</style>
</head>
<body>
<div class="container">

  <div class="topbar">
    <div class="logo">
      <div class="mark"></div>
      <div>
        <div style="font-size:15px; color:var(--muted)">Luxury</div>
        <div style="font-size:18px">Fashion</div>
      </div>
    </div>

    <form class="search" method="get">
      <input type="text" name="q" placeholder="ابحث عن منتج، ماركة، فئة…" value="<?=esc($q)?>">
      <?php // حافظ على بقية الفلاتر عند البحث
      foreach (['category','brand','color','size','material','gender'] as $keep) {
        if (!empty($_GET[$keep]) && is_array($_GET[$keep])) {
          foreach ($_GET[$keep] as $val) echo '<input type="hidden" name="'.$keep.'[]" value="'.esc($val).'">';
        }
      }
      foreach (['price_min','price_max','rating_min','in_stock','on_sale','sort','view','per','page'] as $keep) {
        if (isset($_GET[$keep])) echo '<input type="hidden" name="'.$keep.'" value="'.esc($_GET[$keep]).'">';
      } ?>
      <button type="submit"><i class="fa-solid fa-magnifying-glass"></i> بحث</button>
    </form>

    <div class="actions">
      <div class="pill" id="openPrefs"><i class="fa-solid fa-sliders"></i> تفضيلاتي</div>
      <a class="pill" href="?<?=http_build_query(array_merge($_GET,['view'=>$view==='grid'?'list':'grid']))?>">
        <i class="fa-solid fa-table-cells-large"></i> تبديل العرض
      </a>
      <div class="pill"><i class="fa-regular fa-heart"></i> مفضلتي</div>
      <div class="pill"><i class="fa-solid fa-bag-shopping"></i> السلة</div>
    </div>
  </div>

  <div class="layout">
    <!-- الشريط الجانبي (فلاتر) -->
    <aside class="sidebar">
      <h3><i class="fa-solid fa-filter"></i> فلاتر دقيقة</h3>

      <form method="get">
        <div class="filter-group">
          <div class="small">الفئة</div>
          <div class="chips">
            <?php foreach ($categories as $c): ?>
              <label>
                <input type="checkbox" name="category[]" value="<?=esc($c)?>" <?= in_array($c,$f_cat)?'checked':''?>>
                <span><?=esc($c)?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </div>

        <div class="filter-group">
          <div class="small">الماركة</div>
          <div class="chips">
            <?php foreach ($brands as $c): ?>
              <label>
                <input type="checkbox" name="brand[]" value="<?=esc($c)?>" <?= in_array($c,$f_brand)?'checked':''?>>
                <span><?=esc($c)?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </div>

        <div class="filter-group">
          <div class="small">اللون</div>
          <div class="chips">
            <?php foreach ($colors as $c): ?>
              <label>
                <input type="checkbox" name="color[]" value="<?=esc($c)?>" <?= in_array($c,$f_color)?'checked':''?>>
                <span><?=esc($c)?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </div>

        <div class="filter-group">
          <div class="small">المقاس</div>
          <div class="chips">
            <?php foreach ($sizes as $c): ?>
              <label>
                <input type="checkbox" name="size[]" value="<?=esc($c)?>" <?= in_array($c,$f_size)?'checked':''?>>
                <span><?=esc($c)?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </div>

        <div class="filter-group">
          <div class="small">الخامة</div>
          <div class="chips">
            <?php foreach ($materials as $c): ?>
              <label>
                <input type="checkbox" name="material[]" value="<?=esc($c)?>" <?= in_array($c,$f_material)?'checked':''?>>
                <span><?=esc($c)?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </div>

        <div class="filter-group">
          <div class="small">الجنس</div>
          <div class="chips">
            <?php foreach ($genders as $c): ?>
              <label>
                <input type="checkbox" name="gender[]" value="<?=esc($c)?>" <?= in_array($c,$f_gender)?'checked':''?>>
                <span><?=esc($c)?></span>
              </label>
            <?php endforeach; ?>
          </div>
        </div>

        <div class="filter-group">
          <div class="small">السعر (دولار)</div>
          <div class="range">
            <input type="number" step="0.5" name="price_min" placeholder="من" value="<?= esc($price_min ?? '') ?>">
            <input type="number" step="0.5" name="price_max" placeholder="إلى" value="<?= esc($price_max ?? '') ?>">
          </div>
        </div>

        <div class="filter-group">
          <div class="small">أدنى تقييم</div>
          <div class="range">
            <input type="number" step="0.1" min="0" max="5" name="rating_min" placeholder="0 - 5" value="<?= esc($rating_min ?? '') ?>">
          </div>
          <div class="chips" style="margin-top:8px">
            <label><input type="checkbox" name="in_stock" <?= $in_stock?'checked':'' ?>><span>المتوفر فقط</span></label>
            <label><input type="checkbox" name="on_sale" <?= $on_sale?'checked':'' ?>><span>العروض فقط</span></label>
          </div>
        </div>

        <button class="submit" type="submit"><i class="fa-solid fa-filter"></i> تطبيق الفلاتر</button>
        <a class="reset" href="inventory.php"><i class="fa-solid fa-rotate-right"></i> مسح الكل</a>
      </form>
    </aside>

    <!-- المنطقة الرئيسية -->
    <main class="main">
      <div class="toolbar">
        <div class="left">
          <div class="badge-soft"><?= number_format($total) ?> منتج</div>
          <form method="get" style="display:flex; gap:8px; align-items:center">
            <?php foreach($_GET as $k=>$v){ if(in_array($k,['sort','per','page'])) continue;
              if(is_array($v)){ foreach($v as $vv) echo '<input type="hidden" name="'.esc($k).'[]" value="'.esc($vv).'">'; }
              else echo '<input type="hidden" name="'.esc($k).'" value="'.esc($v).'">';
            } ?>
            <select class="select" name="sort" onchange="this.form.submit()">
              <option value="newest"     <?=$sort==='newest'?'selected':''?>>الأحدث</option>
              <option value="price_asc"  <?=$sort==='price_asc'?'selected':''?>>السعر: الأقل فالأعلى</option>
              <option value="price_desc" <?=$sort==='price_desc'?'selected':''?>>السعر: الأعلى فالأقل</option>
              <option value="popular"    <?=$sort==='popular'?'selected':''?>>الأكثر شعبية</option>
              <option value="rating_desc"<?=$sort==='rating_desc'?'selected':''?>>الأعلى تقييماً</option>
            </select>
            <select class="select" name="per" onchange="this.form.submit()">
              <?php foreach([12,24,36,48,60] as $n): ?>
                <option value="<?=$n?>" <?=$per===$n?'selected':''?>>عرض <?=$n?></option>
              <?php endforeach; ?>
            </select>
            <input type="hidden" name="page" value="1">
          </form>
        </div>
        <div class="right">
          <div class="view-toggle">
            <a href="?<?=http_build_query(array_merge($_GET,['view'=>'grid']))?>">
              <button class="<?= $view==='grid'?'active':''?>"><i class="fa-solid fa-grid-2"></i> شبكة</button>
            </a>
            <a href="?<?=http_build_query(array_merge($_GET,['view'=>'list']))?>">
              <button class="<?= $view==='list'?'active':''?>"><i class="fa-solid fa-list"></i> قائمة</button>
            </a>
          </div>
        </div>
      </div>

      <!-- شبكة المنتجات -->
      <div class="grid <?=$view==='grid'?'grid-view':'list-view'?>">
        <?php if (!$items): ?>
          <div class="card" style="padding:22px; grid-column:1/-1; text-align:center">
            لا توجد نتائج مطابقة لفلاترك. جرّب توسيع البحث أو مسح بعض الفلاتر.
          </div>
        <?php else: foreach($items as $p): ?>
          <div class="card">
            <div class="img" style="background-image:url('<?=esc($p['image_url'])?>')"></div>
            <?php if (!empty($p['on_sale'])): ?><div class="badge">خصم</div><?php endif; ?>
            <div class="fav" title="أضف للمفضلة"><i class="fa-regular fa-heart"></i></div>
            <div class="body">
              <div class="name"><?=esc($p['name'])?></div>
              <div class="meta">
                <span class="badge-soft"><?=esc($p['brand'])?></span>
                <span>•</span>
                <span><?=esc($p['category'])?></span>
                <span>•</span>
                <span style="color:var(--muted)"><?=esc($p['color'])?> · <?=esc($p['size'])?></span>
              </div>
              <div class="price">
                <div class="now">$<?= number_format($p['price'],2) ?></div>
                <?php if (!empty($p['old_price'])): ?><div class="old">$<?= number_format($p['old_price'],2) ?></div><?php endif; ?>
              </div>
              <div class="actions-row">
                <div class="rating" title="تقييم">
                  <?php
                    $stars = floor($p['rating']);
                    $half  = ($p['rating'] - $stars) >= 0.5;
                    for($i=0;$i<$stars;$i++) echo '<i class="fa-solid fa-star"></i>';
                    if ($half) echo '<i class="fa-regular fa-star-half-stroke"></i>';
                    for($i=0; $i<5 - $stars - ($half?1:0); $i++) echo '<i class="fa-regular fa-star"></i>';
                  ?>
                  <span class="small"><?=number_format($p['rating'],1)?></span>
                </div>
                <a class="btn" href="#"><i class="fa-regular fa-eye"></i> نظرة سريعة</a>
                <a class="btn primary" href="#"><i class="fa-solid fa-cart-plus"></i> أضف للسلة</a>
              </div>
            </div>
          </div>
        <?php endforeach; endif; ?>
      </div>

      <!-- ترقيم الصفحات -->
      <?php if ($total_pages>1): ?>
      <div class="pager">
        <a href="<?=page_url(max(1,$page-1))?>"><i class="fa-solid fa-angles-right"></i></a>
        <?php
          $start = max(1, $page-2);
          $end   = min($total_pages, $page+2);
          for($p=$start;$p<=$end;$p++):
        ?>
          <a class="<?= $p==$page?'active':''?>" href="<?=page_url($p)?>"><?=$p?></a>
        <?php endfor; ?>
        <a href="<?=page_url(min($total_pages,$page+1))?>"><i class="fa-solid fa-angles-left"></i></a>
      </div>
      <?php endif; ?>

      <!-- مقترحات حسب التفضيلات -->
      <div class="section">
        <h2><i class="fa-regular fa-gem"></i> مختارات مُفصّلة لك</h2>
        <div class="small">اعتماداً على تفضيلاتك: 
          <?php $pf=$_SESSION['prefs']; ?>
          <span class="badge-soft"><?= esc($pf['gender'] ?? 'يونيسكس') ?></span>
          <?php if(!empty($pf['brands'])): ?><span class="badge-soft">ماركات: <?=esc(implode('، ', $pf['brands']))?></span><?php endif; ?>
          <?php if(!empty($pf['colors'])): ?><span class="badge-soft">ألوان: <?=esc(implode('، ', $pf['colors']))?></span><?php endif; ?>
          <?php if(!empty($pf['sizes'])): ?><span class="badge-soft">مقاسات: <?=esc(implode('، ', $pf['sizes']))?></span><?php endif; ?>
        </div>
        <div class="recs">
          <?php foreach($recs as $p): ?>
            <a class="card" href="#" title="<?=esc($p['name'])?>">
              <div class="img" style="background-image:url('<?=esc($p['image_url'])?>')"></div>
              <?php if (!empty($p['on_sale'])): ?><div class="badge">خصم</div><?php endif; ?>
              <div class="body">
                <div class="name" style="font-size:14px"><?=esc($p['name'])?></div>
                <div class="price">
                  <div class="now" style="font-size:14px">$<?= number_format($p['price'],2) ?></div>
                  <?php if (!empty($p['old_price'])): ?><div class="old" style="font-size:12px">$<?= number_format($p['old_price'],2) ?></div><?php endif; ?>
                </div>
              </div>
            </a>
          <?php endforeach; ?>
        </div>
      </div>

    </main>
  </div>
</div>

<!-- Drawer: تفضيلاتي -->
<div class="drawer" id="prefsDrawer" aria-hidden="true">
  <div class="panel">
    <div style="display:flex; align-items:center; justify-content:space-between">
      <h3><i class="fa-solid fa-sliders"></i> ضبط تفضيلاتي</h3>
      <button onclick="toggleDrawer(false)" class="btn" style="width:auto"><i class="fa-solid fa-xmark"></i> إغلاق</button>
    </div>
    <div class="small">اختر المقاسات، الألوان، الماركات، والجنس المفضل. سيتم تخصيص المقترحات بناءً عليها.</div>
    <div class="hr"></div>

    <div class="row">
      <div>
        <div class="small">المقاسات</div>
        <div class="chips" id="prefSizes"><?php foreach($sizes as $s): ?>
          <label><input type="checkbox" value="<?=esc($s)?>"><span><?=esc($s)?></span></label>
        <?php endforeach; ?></div>
      </div>
      <div>
        <div class="small">الألوان</div>
        <div class="chips" id="prefColors"><?php foreach($colors as $c): ?>
          <label><input type="checkbox" value="<?=esc($c)?>"><span><?=esc($c)?></span></label>
        <?php endforeach; ?></div>
      </div>
      <div>
        <div class="small">الماركات</div>
        <div class="chips" id="prefBrands"><?php foreach($brands as $b): ?>
          <label><input type="checkbox" value="<?=esc($b)?>"><span><?=esc($b)?></span></label>
        <?php endforeach; ?></div>
      </div>
      <div>
        <div class="small">الجنس</div>
        <div class="chips" id="prefGender">
          <?php foreach($genders as $g): ?>
            <label><input type="radio" name="g" value="<?=esc($g)?>"><span><?=esc($g)?></span></label>
          <?php endforeach; ?>
        </div>
      </div>
    </div>

    <button class="save" id="savePrefs"><i class="fa-solid fa-floppy-disk"></i> حفظ التفضيلات</button>
    <div class="small">يتم الحفظ في الجلسة والكوكيز لمدة 30 يوماً.</div>
  </div>
</div>

<script>
// تفعيل/إغلاق درج التفضيلات
const drawer = document.getElementById('prefsDrawer');
function toggleDrawer(show){ drawer.classList.toggle('active', !!show); drawer.setAttribute('aria-hidden', show?'false':'true'); }
document.getElementById('openPrefs').addEventListener('click', ()=>toggleDrawer(true));
drawer.addEventListener('click', (e)=>{ if(e.target===drawer) toggleDrawer(false) });

// تحميل التفضيلات من PHP إلى الواجهة
const currentPrefs = <?= json_encode($_SESSION['prefs'], JSON_UNESCAPED_UNICODE) ?>;
function hydratePrefs(){
  for (const val of (currentPrefs.sizes||[]))  [...document.querySelectorAll('#prefSizes input')].find(i=>i.value===val)?.setAttribute('checked','');
  for (const val of (currentPrefs.colors||[])) [...document.querySelectorAll('#prefColors input')].find(i=>i.value===val)?.setAttribute('checked','');
  for (const val of (currentPrefs.brands||[])) [...document.querySelectorAll('#prefBrands input')].find(i=>i.value===val)?.setAttribute('checked','');
  if (currentPrefs.gender)  [...document.querySelectorAll('#prefGender input')].find(i=>i.value===currentPrefs.gender)?.setAttribute('checked','');
}
hydratePrefs();

// حفظ التفضيلات (AJAX)
document.getElementById('savePrefs').addEventListener('click', async ()=>{
  const sizes  = [...document.querySelectorAll('#prefSizes input:checked')].map(i=>i.value);
  const colors = [...document.querySelectorAll('#prefColors input:checked')].map(i=>i.value);
  const brands = [...document.querySelectorAll('#prefBrands input:checked')].map(i=>i.value);
  const gender = (document.querySelector('#prefGender input:checked')||{}).value || null;

  const res = await fetch('?action=save_prefs', {
    method:'POST',
    headers:{'Content-Type':'application/json'},
    body: JSON.stringify({sizes, colors, brands, gender})
  });
  const data = await res.json().catch(()=>({ok:false}));
  if (data.ok) {
    toggleDrawer(false);
    // تنبيه لطيف
    const note = document.createElement('div');
    note.textContent = 'تم حفظ التفضيلات وتحديث المقترحات ✨';
    note.style.cssText = 'position:fixed;bottom:18px;left:18px;background:rgba(199,162,84,.15);border:1px solid rgba(199,162,84,.4);padding:10px 14px;border-radius:12px;backdrop-filter:blur(6px);z-index:1000';
    document.body.appendChild(note);
    setTimeout(()=>location.reload(), 600);
  }
});
</script>

</body>
</html>
