<?php
require_once __DIR__ . '/../../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ===== Helpers ===== */
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function money($n){ return number_format((float)$n, 2); }
function clean_str(?string $s, int $max=255): ?string{ $s=trim((string)$s); if($s==='') return null; return mb_substr($s,0,$max); }
function hasColumn(PDO $pdo, string $table, string $col): bool{
  try{
    $st=$pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name=? AND column_name=?");
    $st->execute([$table,$col]); return (int)$st->fetchColumn()>0;
  }catch(Throwable $e){ return false; }
}
function tableExists(PDO $pdo, string $table): bool{
  try{
    $st=$pdo->prepare("SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name=?");
    $st->execute([$table]); return (int)$st->fetchColumn()>0;
  }catch(Throwable $e){ return false; }
}
function round005(float $amount): array{
  $rounded = round($amount / 0.05) * 0.05;
  $rounded = round($rounded + 1e-9, 2);
  return [$rounded, $rounded - $amount];
}

/* ===== CSRF ===== */
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf_token'];

/* ===== الصلاحيات المبسطة ===== */
$canChangePrice = !empty($_SESSION['can_change_price']); // اربطها بنظام صلاحياتك

/* ===== إنشاء الجداول عند الحاجة ===== */
$pdo->exec("
CREATE TABLE IF NOT EXISTS sales (
  id INT AUTO_INCREMENT PRIMARY KEY,
  branch_id INT NOT NULL,
  customer_id INT NULL,
  status ENUM('draft','confirmed','paid','cancelled') NOT NULL DEFAULT 'paid',
  sale_date DATE NOT NULL,
  invoice_no VARCHAR(40) NULL,
  currency_code VARCHAR(10) NOT NULL DEFAULT 'EGP',
  subtotal DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  discount_type ENUM('fixed','percent') NOT NULL DEFAULT 'fixed',
  discount_value DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  tax_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  tax_amount DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  service_fee DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  rounding_adjust DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  total DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  paid DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  notes TEXT NULL,
  zone_id INT NULL,
  deleted_at DATETIME NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS sale_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  sale_id INT NOT NULL,
  variant_id INT NOT NULL,
  description VARCHAR(255) NULL,
  qty INT NOT NULL DEFAULT 1,
  unit_price_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  line_discount DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  tax_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  line_total_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  line_tax_extracted DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  CONSTRAINT fk_pos_si_sale FOREIGN KEY (sale_id) REFERENCES sales(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS sale_payments (
  id INT AUTO_INCREMENT PRIMARY KEY,
  sale_id INT NOT NULL,
  method VARCHAR(40) NOT NULL,
  amount DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  ref_no VARCHAR(80) NULL,
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_pos_sp_sale FOREIGN KEY (sale_id) REFERENCES sales(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS variant_branch_prices (
  variant_id INT NOT NULL,
  branch_id INT NOT NULL,
  price_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  PRIMARY KEY (variant_id, branch_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

/* ===== كشف جدول الفروع (Branches أو branches) ===== */
$branchesTable = tableExists($pdo,'Branches') ? 'Branches' : (tableExists($pdo,'branches') ? 'branches' : null);
$BR_ID_COL='id'; $BR_NAME_COL='name';
if ($branchesTable){
  foreach (['id','branch_id','BranchID'] as $c) if (hasColumn($pdo,$branchesTable,$c)) { $BR_ID_COL=$c; break; }
  foreach (['name','branch_name','BranchName'] as $c) if (hasColumn($pdo,$branchesTable,$c)) { $BR_NAME_COL=$c; break; }
}
$branches=[];
if ($branchesTable){
  $branches = $pdo->query("SELECT `$BR_ID_COL` AS id, `$BR_NAME_COL` AS name FROM `$branchesTable` ORDER BY `$BR_NAME_COL`")->fetchAll(PDO::FETCH_KEY_PAIR);
}

/* ===== إعدادات الضريبة ===== */
$settings = ['vat_rate'=>14.00];
try{
  $row = $pdo->query("SELECT * FROM settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);
  if ($row){
    foreach (['vat_rate','VAT','vat','tax_rate','TaxRate'] as $k){
      if (isset($row[$k])) { $settings['vat_rate'] = (float)$row[$k]; break; }
    }
  }
}catch(Throwable $e){}

/* ===== مساعدات أسعار/مخزون ===== */
function get_variant_price(PDO $pdo, int $variant_id, int $branch_id): float {
  $st=$pdo->prepare("SELECT price_incl FROM variant_branch_prices WHERE variant_id=? AND branch_id=?");
  $st->execute([$variant_id,$branch_id]); $p=$st->fetchColumn();
  if ($p!==false) return (float)$p;
  foreach (['price_incl','price','sale_price','Price'] as $c){
    if (hasColumn($pdo,'product_variants',$c)){
      $st=$pdo->prepare("SELECT `$c` FROM product_variants WHERE id=?");
      $st->execute([$variant_id]); $x=$st->fetchColumn();
      if ($x!==false) return (float)$x;
    }
  }
  return 0.00;
}
function last_purchase_cost(PDO $pdo, int $variantId): float {
  try{
    $st=$pdo->prepare("SELECT unit_cost FROM purchase_items WHERE variant_id=? ORDER BY id DESC LIMIT 1");
    $st->execute([$variantId]); $x=$st->fetchColumn();
    return $x!==false ? (float)$x : 0.0;
  }catch(Throwable $e){ return 0.0; }
}

/* ===== Endpoints خفيفة ===== */
if (isset($_GET['searchVariants'])) {
  header('Content-Type: application/json; charset=utf-8');
  $q = trim($_GET['q'] ?? '');
  if ($q===''){ echo json_encode(['ok'=>true,'items'=>[]]); exit; }
  $st=$pdo->prepare("
    SELECT v.id, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS label
    FROM product_variants v
    JOIN products p ON p.id=v.product_id
    WHERE v.deleted_at IS NULL
      AND (v.sku LIKE ? OR p.name_ar LIKE ? OR IFNULL(p.name_en,'') LIKE ?)
    ORDER BY p.name_ar, v.id
    LIMIT 25
  ");
  $like="%$q%"; $st->execute([$like,$like,$like]);
  echo json_encode(['ok'=>true,'items'=>$st->fetchAll(PDO::FETCH_ASSOC)], JSON_UNESCAPED_UNICODE); exit;
}
if (isset($_GET['variantInfo'])) {
  header('Content-Type: application/json; charset=utf-8');
  $vid=(int)($_GET['variant_id'] ?? 0);
  $bid=(int)($_GET['branch_id'] ?? 0);
  if ($vid<=0 || $bid<=0){ echo json_encode(['ok'=>false]); exit; }
  $st=$pdo->prepare("SELECT COALESCE(quantity,0) FROM product_variant_stock WHERE variant_id=? AND branch_id=?");
  $st->execute([$vid,$bid]); $qty=(int)($st->fetchColumn() ?: 0);
  $price=get_variant_price($pdo,$vid,$bid);
  echo json_encode(['ok'=>true,'stock'=>$qty,'price_incl'=>$price,'vat_rate'=>(float)$settings['vat_rate']]); exit;
}

/* ===== حسابات ===== */
function calc_sale_totals(array $items, string $discount_type, float $discount_value, float $vat_rate): array {
  $subtotal_incl = 0.0;
  $tax_extracted = 0.0;
  foreach ($items as $it){
    $qty = max(0,(int)$it['qty']);
    $unit_incl = max(0.0,(float)$it['unit_price_incl']);
    $ld = max(0.0,(float)$it['line_discount']);
    $line_incl = max(0.0, ($qty*$unit_incl) - $ld);
    $subtotal_incl += $line_incl;

    $rate = isset($it['tax_rate']) ? (float)$it['tax_rate'] : $vat_rate;
    $net  = $rate>0 ? ($line_incl/(1+$rate/100)) : $line_incl;
    $tax_extracted += ($line_incl - $net);
  }
  $hdrDisc = ($discount_type==='percent') ? ($subtotal_incl*($discount_value/100)) : $discount_value;
  $hdrDisc = max(0.0,$hdrDisc);
  $after = max(0.0, $subtotal_incl - $hdrDisc);
  [$rounded, $adj] = round005($after);
  return [
    'subtotal_incl'=>$subtotal_incl,
    'tax_extracted'=>$tax_extracted,
    'header_discount'=>$hdrDisc,
    'total'=>$rounded,
    'rounding_adjust'=>$adj
  ];
}

/* ===== خصم مخزون + حركة ===== */
function apply_sale_stock(PDO $pdo, int $sale_id, int $branch_id){
  $st=$pdo->prepare("SELECT variant_id, qty FROM sale_items WHERE sale_id=? ORDER BY id");
  $st->execute([$sale_id]);
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $it){
    $vid=(int)$it['variant_id']; $q=(int)$it['qty'];
    $pdo->prepare("INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
                   VALUES (?,?,?,0)
                   ON DUPLICATE KEY UPDATE quantity = quantity - VALUES(quantity)")
        ->execute([$vid,$branch_id,$q]);
    $uc = last_purchase_cost($pdo,$vid);
    $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                   VALUES (?,?,?,?, 'sale', ?)")->execute([$vid,$branch_id,-$q,$uc,$sale_id]);
  }
}

/* ===== طباعة حراري ===== */
function load_sale(PDO $pdo, int $id){
  $h = $pdo->prepare("SELECT * FROM sales WHERE id=?");
  $h->execute([$id]); $header=$h->fetch(PDO::FETCH_ASSOC);
  if (!$header) return [null,[]];
  $i = $pdo->prepare("
    SELECT si.*, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM sale_items si
    JOIN product_variants v ON v.id=si.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE si.sale_id=?
    ORDER BY si.id
  ");
  $i->execute([$id]); $items=$i->fetchAll(PDO::FETCH_ASSOC);
  return [$header,$items];
}
if (isset($_GET['print']) && $_GET['print']!=='') {
  $id=(int)$_GET['print'];
  [$h,$it]=load_sale($pdo,$id);
  if (!$h){ http_response_code(404); echo "Not found"; exit; }
  // بيانات المتجر/الفرع
  $store = $pdo->query("SELECT * FROM storeinfo LIMIT 1")->fetch(PDO::FETCH_ASSOC) ?: [];
  // اسم الفرع
  $branchName = '';
  if ($branchesTable){
    $bn=$pdo->prepare("SELECT `$BR_NAME_COL` FROM `$branchesTable` WHERE `$BR_ID_COL`=?");
    $bn->execute([(int)$h['branch_id']]); $branchName=$bn->fetchColumn();
  }
  ?><!doctype html>
  <html lang="ar" dir="rtl"><head><meta charset="utf-8"><title>فاتورة #<?= (int)$id ?></title>
  <style>
    @media print { @page{ size: 80mm auto; margin: 5mm; } }
    body{font-family:"Tajawal",Tahoma,Arial,sans-serif; width:72mm; margin:0 auto; color:#000}
    .c{text-align:center} .r{text-align:right}
    .row{display:flex;justify-content:space-between}
    hr{border:0;border-top:1px dashed #000;margin:6px 0}
    table{width:100%;border-collapse:collapse} th,td{padding:2px 0;font-size:12px}
    th{border-bottom:1px solid #000}
    .back-btn{
  background: rgba(255,255,255,.85);
  color:#1f2937;
  border:0;
  border-radius:999px;
  padding:6px 10px;
  font-weight:800;
  text-decoration:none
}
.back-btn:hover{background:#fff; color:#111; text-decoration:none}

  </style>
  </head>
  <body>
  <div class="c">
    <div style="font-size:16px;font-weight:700"><?= esc($store['store_name'] ?? $store['name'] ?? 'المتجر') ?></div>
    <div><?= esc($branchName ?: '-') ?></div>
    <div style="font-size:11px"><?= esc($store['address'] ?? '-') ?></div>
    <div style="font-size:11px">ض.ق: <?= esc($store['vat_number'] ?? $store['tax_id'] ?? '-') ?></div>
  </div>
  <hr>
  <div style="font-size:12px">فاتورة #: <b><?= (int)$id ?></b> — التاريخ: <?= esc($h['sale_date']) ?></div>
  <table>
    <thead><tr><th class="r">الصنف</th><th>كمية</th><th>سعر</th><th>الإجمالي</th></tr></thead>
    <tbody>
    <?php foreach ($it as $row): ?>
      <tr>
        <td class="r"><?= esc(($row['product_name'] ?? '').' | '.($row['sku'] ?? '')) ?></td>
        <td class="c"><?= (int)$row['qty'] ?></td>
        <td class="c"><?= money($row['unit_price_incl']) ?></td>
        <td class="c"><?= money($row['line_total_incl']) ?></td>
      </tr>
    <?php endforeach; ?>
    </tbody>
  </table>
  <hr>
  <div class="row"><div>إجمالي (شامل) قبل الخصم:</div><div><?= money($h['subtotal']) ?></div></div>
  <div class="row"><div>خصم:</div><div><?= money($h['discount_value']) ?> <?= $h['discount_type']==='percent'?'%':'' ?></div></div>
  <div class="row"><div>تقريب:</div><div><?= money($h['rounding_adjust']) ?></div></div>
  <div class="row" style="font-weight:700"><div>الإجمالي:</div><div><?= money($h['total']) ?> <?= esc($h['currency_code']) ?></div></div>
  <div class="row"><div>مدفوع:</div><div><?= money($h['paid']) ?></div></div>
  <div class="row"><div>متبقي:</div><div><?= money((float)$h['total'] - (float)$h['paid']) ?></div></div>
  <hr>
  <div class="c" style="font-size:11px">شكرًا لزيارتكم 🌟</div>
  <script>window.print()</script>
  </body></html><?php
  exit;
}

/* ===== Checkout (حفظ كـ مدفوع) ===== */
if ($_SERVER['REQUEST_METHOD']==='POST' && ($_POST['action'] ?? '')==='checkout') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error='طلب غير صالح (CSRF).';
  } else {
    try{
      $branch_id = (int)($_POST['branch_id'] ?? 0);
      $sale_date = $_POST['sale_date'] ?: date('Y-m-d');
      $currency  = clean_str($_POST['currency_code'] ?? 'EGP', 10);
      $vat_rate  = (float)($_POST['tax_rate'] ?? $settings['vat_rate']);
      $discount_type = $_POST['discount_type'] ?? 'fixed';
      $discount_value= (float)($_POST['discount_value'] ?? 0);
      $notes = clean_str($_POST['notes'] ?? '', 2000);
      $pay_method = clean_str($_POST['pay_method'] ?? 'Cash', 40);
      $pay_amount = (float)($_POST['pay_amount'] ?? 0);

      if ($branch_id<=0) throw new RuntimeException('اختر فرعًا.');

      // بنود
      $var_ids = array_map('intval', $_POST['item_variant_id'] ?? []);
      $qty_arr = array_map('intval', $_POST['item_qty'] ?? []);
      $price_arr = array_map('floatval', $_POST['item_unit_price'] ?? []);
      $disc_arr  = array_map('floatval', $_POST['item_line_discount'] ?? []);
      $items=[];
      for($i=0;$i<count($var_ids);$i++){
        if ($var_ids[$i]<=0) continue;
        $qty = max(0,(int)($qty_arr[$i] ?? 0));
        if ($qty===0) continue;
        $unit = max(0.0,(float)($price_arr[$i] ?? 0));
        $ld   = max(0.0,(float)($disc_arr[$i] ?? 0));
        if (!$canChangePrice){
          $unit = get_variant_price($pdo,(int)$var_ids[$i],$branch_id);
        }
        $items[] = [
          'variant_id'=>$var_ids[$i], 'qty'=>$qty,
          'unit_price_incl'=>$unit, 'line_discount'=>$ld, 'tax_rate'=>$vat_rate
        ];
      }
      if (!$items) throw new RuntimeException('أضف صنفًا واحدًا على الأقل.');

      // حساب الإجماليات
      $tot = calc_sale_totals($items, $discount_type, $discount_value, $vat_rate);

      // تحذيرات سالب
      $negativeWarnings=[];
      foreach ($items as $it){
        $st=$pdo->prepare("SELECT COALESCE(quantity,0) FROM product_variant_stock WHERE variant_id=? AND branch_id=?");
        $st->execute([(int)$it['variant_id'],$branch_id]);
        $cur=(int)($st->fetchColumn() ?: 0);
        if ($cur - (int)$it['qty'] < 0){
          $negativeWarnings[] = "تحذير: المتغير #".$it['variant_id']." سيُصبح سالب.";
        }
      }

      $pdo->beginTransaction();
      // رأس
      $st=$pdo->prepare("INSERT INTO sales (branch_id, customer_id, status, sale_date, currency_code, subtotal, discount_type, discount_value, tax_rate, tax_amount, service_fee, rounding_adjust, total, paid, notes)
                         VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
      $st->execute([$branch_id,null,'paid',$sale_date,$currency,$tot['subtotal_incl'],$discount_type,$discount_value,$vat_rate,$tot['tax_extracted'],0,$tot['rounding_adjust'],$tot['total'],$tot['total'],$notes]);
      $sid = (int)$pdo->lastInsertId();

      // البنود
      $ins=$pdo->prepare("INSERT INTO sale_items (sale_id, variant_id, description, qty, unit_price_incl, line_discount, tax_rate, line_total_incl, line_tax_extracted)
                          VALUES (?,?,?,?,?,?,?,?,?)");
      foreach($items as $it){
        $qty=$it['qty']; $unit=$it['unit_price_incl']; $ld=$it['line_discount']; $tr=$it['tax_rate'];
        $line_incl = max(0.0, ($qty*$unit)-$ld);
        $net = $tr>0 ? ($line_incl/(1+$tr/100)) : $line_incl;
        $tax = $line_incl - $net;
        $ins->execute([$sid,$it['variant_id'],null,$qty,$unit,$ld,$tr,$line_incl,$tax]);
      }

      // دفعة (قيمة الدفعة لو أقل/أكثر من الإجمالي نضيف كما هي، ونترك المتبقي/الباقي للتقرير)
      if ($pay_amount>0){
        $pdo->prepare("INSERT INTO sale_payments (sale_id, method, amount, ref_no, notes) VALUES (?,?,?,?,?)")
            ->execute([$sid,$pay_method,$pay_amount,null,null]);
        $sum = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM sale_payments WHERE sale_id=?");
        $sum->execute([$sid]); $paid=(float)$sum->fetchColumn();
        $pdo->prepare("UPDATE sales SET paid=? WHERE id=?")->execute([$paid,$sid]);
      }

      // خصم المخزون
      apply_sale_stock($pdo,$sid,$branch_id);

      $pdo->commit();
      // بعد الحفظ: إعادة توجيه للطباعة لو تحب
      header("Location: sale.php?done=$sid");
      exit;

    }catch(Throwable $e){
      if ($pdo->inTransaction()) $pdo->rollBack();
      $error = 'خطأ: '.esc($e->getMessage());
    }
  }
}

/* ===== واجهة ===== */
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>POS بيع سريع</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#fff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc; --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{ background:linear-gradient(90deg,var(--p1),var(--p2),var(--p3),var(--p4)); padding:12px 16px; display:flex; align-items:center; justify-content:space-between; box-shadow:0 6px 24px rgba(0,0,0,.15); }
    .wrap{padding:16px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .num{min-width:90px;text-align:center}
    .neg{background:#f8d7da;border:1px solid #f5c2c7;border-radius:999px;padding:2px 8px}
    .kbd{background:#111;color:#fff;border-radius:6px;padding:.1rem .35rem;font-family:monospace;font-size:.85rem}
    .back-btn{
  background: rgba(255,255,255,.85);
  color:#1f2937;
  border:0;
  border-radius:999px;
  padding:6px 10px;
  font-weight:800;
  text-decoration:none
}
.back-btn:hover{background:#fff; color:#111; text-decoration:none}

  </style>
</head>
<body>
<header>
  <div class="d-flex align-items-center gap-2">
    <strong><i class="fa-solid fa-cash-register me-1"></i> POS بيع سريع</strong>
  </div>
  <div class="d-flex align-items-center gap-2">
    <a href="../dashboard.php" class="back-btn">
      <i class="fa-solid fa-house"></i> لوحة التحكم
    </a>
    <select id="branch" class="form-select form-select-sm" style="min-width:200px">
      <option value="">— اختر فرع —</option>
      <?php foreach ($branches as $bid=>$bn): ?>
        <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
      <?php endforeach; ?>
    </select>
    <button id="darkSwitch" class="btn btn-sm btn-primary">🌙</button>
  </div>
</header>


<div class="wrap">
  <div class="row g-3">
    <div class="col-lg-8">
      <div class="card-soft p-3">
        <div class="d-flex flex-wrap gap-2 align-items-center mb-2">
          <input type="text" id="scan" class="form-control" placeholder="سكان باركود / اكتب SKU أو اسم واضغط Enter" style="min-width:260px">
          <span class="pill">الأسعار شاملة الضريبة (VAT <?= (float)$settings['vat_rate'] ?>%)</span>
          <span class="pill">تقريب لأقرب 0.05</span>
          <span class="pill">الاختصارات: <span class="kbd">Enter</span> إضافة، <span class="kbd">F4</span> دفع، <span class="kbd">Del</span> حذف سطر</span>
        </div>
        <div class="table-responsive">
          <table class="table table-sm align-middle text-center">
            <thead class="table-light">
              <tr>
                <th style="min-width:260px">الصنف</th>
                <th class="num">المخزون</th>
                <th class="num">كمية</th>
                <th class="num">سعر</th>
                <th class="num">خصم</th>
                <th class="num">إجمالي</th>
                <th></th>
              </tr>
            </thead>
            <tbody id="itemsBody"></tbody>
          </table>
        </div>
        <button class="btn btn-outline-primary" id="addEmpty">➕ إضافة سطر يدوي</button>
      </div>
    </div>

    <div class="col-lg-4">
      <div class="card-soft p-3">
        <div class="mb-2">
          <label class="form-label">خصم عام</label>
          <div class="input-group">
            <select id="discount_type" class="form-select" style="max-width:110px">
              <option value="fixed">قيمة</option>
              <option value="percent">%</option>
            </select>
            <input type="number" step="0.01" id="discount_value" class="form-control" value="0">
          </div>
        </div>
        <div class="d-flex justify-content-between"><span>مجموع (شامل):</span><strong id="t_subtotal">0.00</strong></div>
        <div class="d-flex justify-content-between"><span>خصم عام:</span><strong id="t_hdrdisc">0.00</strong></div>
        <div class="d-flex justify-content-between"><span>تقريب:</span><strong id="t_round">0.00</strong></div>
        <hr class="my-2">
        <div class="d-flex justify-content-between fs-5"><span>الإجمالي:</span><strong id="t_total">0.00</strong></div>
        <div class="mt-3 d-grid gap-2">
          <button class="btn btn-primary" id="payBtn"><i class="fa-solid fa-money-bill-wave"></i> دفع (F4)</button>
          <?php if (isset($_GET['done'])): ?>
            <a class="btn btn-outline-dark" target="_blank" href="?print=<?= (int)$_GET['done'] ?>"><i class="fa-solid fa-print"></i> طباعة آخر فاتورة</a>
          <?php endif; ?>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Payment Modal -->
<div class="modal fade" id="payModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-credit-card me-2"></i> دفع</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body" id="payForm">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="checkout">
      <input type="hidden" name="sale_date" value="<?= date('Y-m-d') ?>">
      <input type="hidden" name="currency_code" value="EGP">
      <input type="hidden" name="tax_rate" id="f_tax_rate" value="<?= (float)$settings['vat_rate'] ?>">
      <input type="hidden" name="discount_type" id="f_discount_type">
      <input type="hidden" name="discount_value" id="f_discount_value">
      <input type="hidden" name="notes" value="">
      <div class="mb-2">
        <label class="form-label">الفرع *</label>
        <select class="form-select" name="branch_id" id="f_branch" required>
          <option value="">— اختر —</option>
          <?php foreach ($branches as $bid=>$bn): ?>
            <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="d-flex justify-content-between">
        <div>الإجمالي</div><div id="pay_total" class="fw-bold">0.00</div>
      </div>
      <div class="mb-2">
        <label class="form-label">طريقة الدفع</label>
        <select class="form-select" name="pay_method" id="pay_method">
          <option>Cash</option>
          <option>Card</option>
          <option>Wallet</option>
          <option>Transfer</option>
        </select>
      </div>
      <div class="mb-3">
        <label class="form-label">المبلغ المدفوع</label>
        <input type="number" step="0.01" class="form-control" name="pay_amount" id="pay_amount" value="0">
        <div class="mt-2 d-flex gap-2 flex-wrap">
          <button type="button" class="btn btn-outline-secondary btn-sm quick" data-val="Exact">المبلغ كامل</button>
          <button type="button" class="btn btn-outline-secondary btn-sm quick" data-val="50">+50</button>
          <button type="button" class="btn btn-outline-secondary btn-sm quick" data-val="100">+100</button>
          <button type="button" class="btn btn-outline-secondary btn-sm quick" data-val="200">+200</button>
        </div>
      </div>

      <!-- عناصر السلة تُحقن هنا فور الفتح -->
      <div id="itemsHidden"></div>

      <div class="text-center">
        <button class="btn btn-primary px-4"><i class="fa-solid fa-check"></i> تأكيد واصدار فاتورة</button>
      </div>
    </form>
  </div></div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark mode
  const body=document.body, darkSwitch=document.getElementById('darkSwitch');
  if (localStorage.getItem('darkMode')==='true'){ body.classList.add('dark'); darkSwitch.textContent='☀️'; }
  darkSwitch?.addEventListener('click', ()=>{ body.classList.toggle('dark'); const d=body.classList.contains('dark'); localStorage.setItem('darkMode', d); darkSwitch.textContent=d?'☀️':'🌙'; });

  const itemsBody = document.getElementById('itemsBody');
  const scan = document.getElementById('scan');
  const branchSel = document.getElementById('branch');

  const canChangePrice = <?= $canChangePrice ? 'true' : 'false' ?>;
  const defaultVat = parseFloat(document.getElementById('f_tax_rate').value||0);

  // إضافة صف فارغ
  document.getElementById('addEmpty')?.addEventListener('click', ()=> addRow({}));

  // سكان/بحث
  scan?.addEventListener('keydown', async (e)=>{
    if (e.key!=='Enter') return;
    e.preventDefault();
    const q = scan.value.trim(); if (!q) return;
    const res = await fetch(location.href + (location.search ? '&':'?') + 'searchVariants=1&q='+encodeURIComponent(q));
    const data = await res.json();
    let match = null;
    if (data && data.items) {
      match = data.items.find(x=> (x.sku||'').toLowerCase()===q.toLowerCase()) || data.items[0];
    }
    if (!match){ alert('غير موجود'); return; }
    // لو موجود في السلة، زوّد الكمية
    const exist = [...document.querySelectorAll('#itemsBody tr')].find(tr=> tr.querySelector('[name="item_variant_id[]"]')?.value == match.id);
    if (exist){
      const qEl = exist.querySelector('[name="item_qty[]"]'); qEl.value = (parseInt(qEl.value||0)+1); calcLine(exist);
    } else {
      addRow({variant_id:match.id, label: `${match.id} | ${match.label} | ${match.sku}`});
    }
    scan.value='';
  });

  function addRow(opt){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" placeholder="اكتب SKU/اسم واضغط Enter" value="${opt.label? escapeHtml(opt.label): ''}" onkeydown="variantKey(this,event)">
        <input type="hidden" name="item_variant_id[]" value="${opt.variant_id||''}">
      </td>
      <td class="num"><span class="pill stock-pill">?</span></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="${opt.qty||1}" oninput="calcLine(this.closest('tr'))"></td>
      <td><input type="number" step="0.01" name="item_unit_price[]" class="form-control text-center" value="${opt.price||0}" ${canChangePrice? '' : 'readonly'} oninput="calcLine(this.closest('tr'))"></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="${opt.disc||0}" oninput="calcLine(this.closest('tr'))"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
    if (opt.variant_id) refreshVariantInfo(tr, opt.variant_id);
    calcLine(tr);
  }

  async function refreshVariantInfo(tr, vid){
    const bid = branchSel.value||'';
    if (!bid){ tr.querySelector('.stock-pill').textContent='-'; return; }
    const res = await fetch(location.href + (location.search ? '&':'?') + 'variantInfo=1&variant_id='+encodeURIComponent(vid)+'&branch_id='+encodeURIComponent(bid));
    const d = await res.json();
    if (!d || !d.ok) return;
    tr.querySelector('.stock-pill').textContent = d.stock;
    const priceEl = tr.querySelector('[name="item_unit_price[]"]');
    if (!priceEl.value || parseFloat(priceEl.value)==0){ priceEl.value = (d.price_incl||0).toFixed(2); }
    if (!isNaN(parseFloat(priceEl.value))) calcLine(tr);
  }

  function variantKey(inp, e){
    if (e.key!=='Enter') return;
    e.preventDefault();
    const q = inp.value.trim(); if (!q) return;
    const idFromText = parseInt(q.split('|')[0],10);
    if (!isNaN(idFromText)){ selectVariant(inp, idFromText, q); return; }
    fetch(location.href + (location.search ? '&':'?') + 'searchVariants=1&q='+encodeURIComponent(q))
      .then(r=>r.json()).then(d=>{
        const it=(d.items||[])[0];
        if (!it){ alert('غير موجود'); return; }
        selectVariant(inp, it.id, `${it.id} | ${it.label} | ${it.sku}`);
      });
  }
  function selectVariant(inp, id, display){
    const tr = inp.closest('tr');
    inp.value = display || (id+'');
    tr.querySelector('[name="item_variant_id[]"]').value = id;
    refreshVariantInfo(tr, id);
  }

  function calcLine(tr){
    const qty = parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
    const unit= parseFloat(tr.querySelector('[name="item_unit_price[]"]').value||0);
    const disc= parseFloat(tr.querySelector('[name="item_line_discount[]"]').value||0);
    let line = Math.max(0, qty*unit - disc);
    tr.querySelector('.line-total').textContent = line.toFixed(2);

    // تحذير سالب
    const stockTxt = tr.querySelector('.stock-pill').textContent;
    const stock = parseInt(stockTxt||0);
    if (!isNaN(stock) && qty>stock){
      tr.querySelector('.stock-pill').classList.add('neg');
      tr.querySelector('.stock-pill').title = 'سيصبح المخزون سالب عند الدفع';
    } else {
      tr.querySelector('.stock-pill').classList.remove('neg');
      tr.querySelector('.stock-pill').title = '';
    }
    calcTotals();
  }

  function calcTotals(){
    let subtotal=0;
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      subtotal += parseFloat(tr.querySelector('.line-total').textContent||0);
    });
    const dtype = document.getElementById('discount_type').value;
    const dval  = parseFloat(document.getElementById('discount_value').value||0);
    let hdrDisc = (dtype==='percent') ? (subtotal*(dval/100)) : dval; hdrDisc=Math.max(0,hdrDisc);
    const after = Math.max(0, subtotal - hdrDisc);
    const rounded = Math.round((after/0.05))*0.05;
    const adj = rounded - after;

    document.getElementById('t_subtotal').textContent = subtotal.toFixed(2);
    document.getElementById('t_hdrdisc').textContent  = hdrDisc.toFixed(2);
    document.getElementById('t_round').textContent    = adj.toFixed(2);
    document.getElementById('t_total').textContent    = rounded.toFixed(2);
  }

  // حذف سطر بـ Del
  document.addEventListener('keydown', (e)=>{
    if (e.key==='Delete'){
      const focused = document.activeElement;
      const tr = focused?.closest?.('tr.item-row');
      if (tr){ tr.remove(); calcTotals(); }
    }
    if (e.key==='F4'){ e.preventDefault(); openPay(); }
  });

  // فتح الدفع
  const payModal = new bootstrap.Modal(document.getElementById('payModal'));
  document.getElementById('payBtn')?.addEventListener('click', openPay);
  function openPay(){
    // لازم فرع
    if (!branchSel.value){ alert('اختر الفرع أولًا'); branchSel.focus(); return; }
    // لازم صنف
    if (!document.querySelector('#itemsBody [name="item_variant_id[]"]')){ alert('السلة فارغة'); return; }
    // جهّز hidden inputs
    const box = document.getElementById('itemsHidden'); box.innerHTML='';
    const ids   = [...document.querySelectorAll('[name="item_variant_id[]"]')].map(x=>x.value).filter(Boolean);
    const qtys  = [...document.querySelectorAll('[name="item_qty[]"]')].map(x=>x.value||0);
    const price = [...document.querySelectorAll('[name="item_unit_price[]"]')].map(x=>x.value||0);
    const disc  = [...document.querySelectorAll('[name="item_line_discount[]"]')].map(x=>x.value||0);
    ids.forEach((v,i)=>{
      box.innerHTML += `<input type="hidden" name="item_variant_id[]" value="${escapeHtml(v)}">`;
      box.innerHTML += `<input type="hidden" name="item_qty[]" value="${escapeHtml(qtys[i])}">`;
      box.innerHTML += `<input type="hidden" name="item_unit_price[]" value="${escapeHtml(price[i])}">`;
      box.innerHTML += `<input type="hidden" name="item_line_discount[]" value="${escapeHtml(disc[i])}">`;
    });
    // نسخ خصم عام
    document.getElementById('f_discount_type').value  = document.getElementById('discount_type').value;
    document.getElementById('f_discount_value').value = document.getElementById('discount_value').value;
    // فرع
    document.getElementById('f_branch').value = branchSel.value;
    // الإجمالي
    document.getElementById('pay_total').textContent = document.getElementById('t_total').textContent;
    // أزرار سريعة
    document.querySelectorAll('.quick').forEach(btn=>{
      btn.onclick = ()=>{
        const v = btn.dataset.val;
        const input = document.getElementById('pay_amount');
        if (v==='Exact'){ input.value = document.getElementById('t_total').textContent; }
        else { input.value = (parseFloat(input.value||0) + parseFloat(v||0)).toFixed(2); }
      };
    });
    payModal.show();
  }

  // تغيير الفرع يعيد تحميل المخزون/السعر لكل الأسطر
  branchSel?.addEventListener('change', ()=>{
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      const id = tr.querySelector('[name="item_variant_id[]"]')?.value;
      if (id) refreshVariantInfo(tr, id);
    });
  });

  // وظائف مساعدة
  function escapeHtml(s){ return (s||'').toString().replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
  window.variantKey = variantKey; // make it global
  window.calcLine = calcLine;
</script>
</body>
</html>
