<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
// Helpers صغيرة
function hasColumn(PDO $pdo, string $table, string $col): bool {
    $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = ? AND column_name = ?");
    $st->execute([$table,$col]);
    return (int)$st->fetchColumn() > 0;
  }
  function supplierNameCol(PDO $pdo): string {
    // جرّب أشهر الأسماء الممكنة
    foreach (['name','SupplierName','supplier_name','company_name','CompanyName'] as $c) {
      if (hasColumn($pdo,'suppliers',$c)) return $c;
    }
    // fallback
    return 'name';
  }
  $SUPPLIER_NAME_COL = supplierNameCol($pdo);
  
$success = $error = "";

/* صلاحيات (بسّط حسب نظامك) */
$canEdit = true;

/* CSRF */
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}
$csrf = $_SESSION['csrf_token'];

/* Helpers */
function clean_str(?string $s, int $max=255): ?string {
  $s = trim((string)$s); if ($s==='') return null; return mb_substr($s, 0, $max);
}
function money($n){ return number_format((float)$n, 2); }
function array_sanitize_nums($arr){ return array_map(fn($x)=> (float)$x, (array)$arr); }

/* فروع */
$branches = [];
try {
  $branches = $pdo->query("SELECT BranchID, BranchName FROM Branches ORDER BY BranchName")->fetchAll(PDO::FETCH_KEY_PAIR);
} catch (Throwable $e) { $branches = []; }

/* موردون */
$hasSupActive = hasColumn($pdo,'suppliers','is_active');
$sqlSup = "SELECT id, `{$SUPPLIER_NAME_COL}` AS name FROM suppliers"
        . ($hasSupActive ? " WHERE is_active=1" : "")
        . " ORDER BY `{$SUPPLIER_NAME_COL}`";
$suppliers = $pdo->query($sqlSup)->fetchAll(PDO::FETCH_KEY_PAIR);


/* منتجات/متغيّرات للبحث السريع */
$variants = $pdo->query("
  SELECT v.id,
         CONCAT(p.name_ar, ' | ', IFNULL(p.name_en,'-'), ' | ', v.sku) AS label
  FROM product_variants v
  LEFT JOIN products p ON p.id = v.product_id
  WHERE v.deleted_at IS NULL
  ORDER BY p.name_ar, v.id
  LIMIT 2000
")->fetchAll(PDO::FETCH_ASSOC);

/* تحميل تفاصيل عملية شراء واحدة */
function load_purchase(PDO $pdo, int $id){
  $h = $pdo->prepare("SELECT * FROM purchases WHERE id=?");
  $h->execute([$id]);
  $header = $h->fetch(PDO::FETCH_ASSOC);
  if (!$header) return [null, []];
  $i = $pdo->prepare("
    SELECT pi.*, v.sku,
           CONCAT(p.name_ar, ' | ', IFNULL(p.name_en,'-')) AS product_name
    FROM purchase_items pi
    JOIN product_variants v ON v.id=pi.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE pi.purchase_id=?
    ORDER BY pi.id
  ");
  $i->execute([$id]);
  return [$header, $i->fetchAll(PDO::FETCH_ASSOC)];
}

/* إعادة حساب الإجماليات (سيرفر) */
function calc_totals(array $items, string $discount_type, float $discount_value, float $tax_rate, float $shipping, float $other_fees){
  $subtotal = 0.0;
  foreach ($items as $it) {
    $line = max(0, ($it['ordered_qty'] ?? 0)) * (float)$it['unit_cost'] - (float)$it['line_discount'];
    $line_tax = $line * ((float)($it['line_tax_rate'] ?? 0)/100);
    $subtotal += ($line + $line_tax);
  }
  $header_disc = ($discount_type==='percent') ? ($subtotal * ($discount_value/100)) : $discount_value;
  $after_disc  = max(0, $subtotal - (float)$header_disc);
  $header_tax  = $after_disc * ($tax_rate/100);
  $total = $after_disc + $header_tax + $shipping + $other_fees;
  return [$subtotal, $header_disc, $header_tax, $total];
}

/* تحديث مخزون + دفتر حركة عند الاستلام */
function receive_and_move(PDO $pdo, int $purchase_id, int $branch_id){
  // نحتاج مقارنة received_qty السابقة بالجديدة ثم تسجيل الفروقات
  $rows = $pdo->prepare("SELECT id, variant_id, ordered_qty, received_qty FROM purchase_items WHERE purchase_id=? ORDER BY id");
  $rows->execute([$purchase_id]);
  $items = $rows->fetchAll(PDO::FETCH_ASSOC);
  foreach ($items as $it){
    $id = (int)$it['id'];
    $prev = (int)($GLOBALS['_prev_received'][$id] ?? 0);
    $now  = (int)$it['received_qty'];
    $delta = $now - $prev;
    if ($delta > 0 && $branch_id){
      // آخر تكلفة سطر لهذه البنود (تقريبًا متوسط تكلفة الشراء)
      $costRow = $pdo->prepare("SELECT unit_cost FROM purchase_items WHERE id=?");
      $costRow->execute([$id]);
      $ucost = (float)$costRow->fetchColumn();

      // تحديث مخزون الفرع
      $up = $pdo->prepare("
        INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
        VALUES (?,?,?,0)
        ON DUPLICATE KEY UPDATE quantity = quantity + VALUES(quantity)
      ");
      $up->execute([(int)$it['variant_id'], $branch_id, $delta]);

      // دفتر حركة
      $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                     VALUES (?,?,?,?, 'purchase', ?)")->execute([
        (int)$it['variant_id'], $branch_id, $delta, $ucost, $purchase_id
      ]);
    }
  }
}

/* خريطة received السابقة عند التعديل/الاستلام */
$_prev_received = [];

/* ==== الإجراءت ==== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF).'; 
  } elseif (!$canEdit) {
    $error = 'لا تملك صلاحية التعديل.';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action === 'save_purchase') {
        $id            = ($_POST['id'] ?? '')!=='' ? (int)$_POST['id'] : null;
        $supplier_id   = (int)($_POST['supplier_id'] ?? 0);
        $branch_id     = ($_POST['branch_id'] ?? '')!=='' ? (int)$_POST['branch_id'] : null;
        $status        = $_POST['status'] ?? 'draft';
        $order_date    = $_POST['order_date'] ?: date('Y-m-d');
        $expected_date = $_POST['expected_date'] ?: null;
        $invoice_no    = clean_str($_POST['invoice_no'] ?? '', 80);
        $invoice_date  = $_POST['invoice_date'] ?: null;
        $currency      = clean_str($_POST['currency_code'] ?? 'EGP', 10);
        $discount_type = $_POST['discount_type'] ?? 'fixed';
        $discount_value= (float)($_POST['discount_value'] ?? 0);
        $tax_rate      = (float)($_POST['tax_rate'] ?? 0);
        $shipping      = (float)($_POST['shipping'] ?? 0);
        $other_fees    = (float)($_POST['other_fees'] ?? 0);
        $notes         = clean_str($_POST['notes'] ?? '', 10000);

        // عناصر
        $var_ids       = array_map('intval', $_POST['item_variant_id'] ?? []);
        $desc_arr      = $_POST['item_desc'] ?? [];
        $ord_qty_arr   = array_map('intval', $_POST['item_ordered_qty'] ?? []);
        $rec_qty_arr   = array_map('intval', $_POST['item_received_qty'] ?? []);
        $cost_arr      = array_map('floatval', $_POST['item_unit_cost'] ?? []);
        $line_disc_arr = array_map('floatval', $_POST['item_line_discount'] ?? []);
        $line_tax_arr  = array_map('floatval', $_POST['item_line_tax'] ?? []);

        $items = [];
        for ($i=0; $i<count($var_ids); $i++){
          if ($var_ids[$i] <= 0) continue;
          $ordered = max(0, (int)($ord_qty_arr[$i] ?? 0));
          $received= max(0, (int)($rec_qty_arr[$i] ?? 0));
          $unit    = max(0.0, (float)($cost_arr[$i] ?? 0));
          $ldisc   = max(0.0, (float)($line_disc_arr[$i] ?? 0));
          $ltax    = max(0.0, (float)($line_tax_arr[$i] ?? 0));
          $desc    = clean_str($desc_arr[$i] ?? '', 255);
          if ($ordered===0 && $received===0) continue; // تجاهل سطر فارغ
          $items[] = [
            'variant_id'=>$var_ids[$i],
            'description'=>$desc,
            'ordered_qty'=>$ordered,
            'received_qty'=>$received,
            'unit_cost'=>$unit,
            'line_discount'=>$ldisc,
            'line_tax_rate'=>$ltax
          ];
        }
        if ($supplier_id<=0) throw new RuntimeException('اختر موردًا.');
        if (!$items) throw new RuntimeException('أضف بندًا واحدًا على الأقل.');

        // حساب الإجماليات
        [$subtotal, $header_disc, $header_tax, $total] =
          calc_totals($items, $discount_type, $discount_value, $tax_rate, $shipping, $other_fees);

        // مرفق
        $attachName = null;
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
          $uploadDir = __DIR__ . '/uploads/purchases/';
          if (!is_dir($uploadDir)) @mkdir($uploadDir,0777,true);
          $ext = strtolower(pathinfo($_FILES['attachment']['name'], PATHINFO_EXTENSION));
          $name = time().'_'.bin2hex(random_bytes(4)).'.'.$ext;
          if (move_uploaded_file($_FILES['attachment']['tmp_name'], $uploadDir.$name)) {
            $attachName = $name;
          }
        }

        if ($id) {
          // خزّن received السابق قبل التحديث
          $pr = $pdo->prepare("SELECT id, received_qty FROM purchase_items WHERE purchase_id=?");
          $pr->execute([$id]);
          foreach ($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_received[(int)$r['id']] = (int)$r['received_qty']; }

          $sql = "UPDATE purchases SET supplier_id=:supplier_id, branch_id=:branch_id, status=:status,
                  order_date=:order_date, expected_date=:expected_date, invoice_no=:invoice_no, invoice_date=:invoice_date,
                  currency_code=:currency_code, subtotal=:subtotal, discount_type=:discount_type, discount_value=:discount_value,
                  tax_rate=:tax_rate, shipping=:shipping, other_fees=:other_fees, total=:total, notes=:notes"
                 . ($attachName ? ", attachment=:attachment" : "")
                 . " WHERE id=:id AND deleted_at IS NULL";
          $st = $pdo->prepare($sql);
          $params = [
            ':supplier_id'=>$supplier_id, ':branch_id'=>$branch_id, ':status'=>$status,
            ':order_date'=>$order_date, ':expected_date'=>$expected_date,
            ':invoice_no'=>$invoice_no, ':invoice_date'=>$invoice_date,
            ':currency_code'=>$currency, ':subtotal'=>$subtotal, ':discount_type'=>$discount_type,
            ':discount_value'=>$discount_value, ':tax_rate'=>$tax_rate, ':shipping'=>$shipping,
            ':other_fees'=>$other_fees, ':total'=>$total, ':notes'=>$notes, ':id'=>$id
          ];
          if ($attachName) $params[':attachment']=$attachName;
          $st->execute($params);

          // حدّث البنود: احذف الكل ثم أدخل من جديد (أبسط)
          $pdo->prepare("DELETE FROM purchase_items WHERE purchase_id=?")->execute([$id]);
          $ins = $pdo->prepare("INSERT INTO purchase_items
              (purchase_id, variant_id, description, ordered_qty, received_qty, unit_cost, line_discount, line_tax_rate, line_total)
              VALUES (?,?,?,?,?,?,?,?,?)");
          foreach ($items as $it){
            $line_val = ($it['ordered_qty']*$it['unit_cost']) - $it['line_discount'];
            $line_val += $line_val * ($it['line_tax_rate']/100);
            $ins->execute([$id, $it['variant_id'], $it['description'], $it['ordered_qty'], $it['received_qty'],
              $it['unit_cost'], $it['line_discount'], $it['line_tax_rate'], $line_val]);
          }

          // في حالة الاستلام (كلي/جزئي) حدّث المخزون
          if (in_array($status, ['partial','received'], true) && $branch_id) {
            receive_and_move($pdo, $id, $branch_id);
          }

          $success = '✔️ تم تحديث العملية';

        } else {
          // إنشاء رأس
          $st = $pdo->prepare("INSERT INTO purchases
            (supplier_id, branch_id, status, order_date, expected_date, invoice_no, invoice_date, currency_code,
             subtotal, discount_type, discount_value, tax_rate, shipping, other_fees, total, notes, attachment)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
          $st->execute([
            $supplier_id, $branch_id, $status, $order_date, $expected_date, $invoice_no, $invoice_date, $currency,
            $subtotal, $discount_type, $discount_value, $tax_rate, $shipping, $other_fees, $total, $notes, $attachName
          ]);
          $pid = (int)$pdo->lastInsertId();

          // بنود
          $ins = $pdo->prepare("INSERT INTO purchase_items
              (purchase_id, variant_id, description, ordered_qty, received_qty, unit_cost, line_discount, line_tax_rate, line_total)
              VALUES (?,?,?,?,?,?,?,?,?)");
          foreach ($items as $it){
            $line_val = ($it['ordered_qty']*$it['unit_cost']) - $it['line_discount'];
            $line_val += $line_val * ($it['line_tax_rate']/100);
            $ins->execute([$pid, $it['variant_id'], $it['description'], $it['ordered_qty'], $it['received_qty'],
              $it['unit_cost'], $it['line_discount'], $it['line_tax_rate'], $line_val]);
          }

          // استلام فوري؟ (لو الحالة partial/received)
          if (in_array($status, ['partial','received'], true) && $branch_id) {
            // prev received كلها 0 لأن العملية جديدة
            $_prev_received = [];
            receive_and_move($pdo, $pid, $branch_id);
          }

          $success = '✔️ تم إنشاء العملية';
        }

      } elseif ($action === 'receive_all') {
        $id = (int)($_POST['id'] ?? 0);
        if ($id<=0) throw new RuntimeException('عملية غير معروفة.');
        // خزّن السابق
        $pr = $pdo->prepare("SELECT id, received_qty FROM purchase_items WHERE purchase_id=?");
        $pr->execute([$id]);
        foreach ($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_received[(int)$r['id']] = (int)$r['received_qty']; }
        // اجعل received=ordered لكل البنود
        $pdo->prepare("UPDATE purchase_items SET received_qty = ordered_qty WHERE purchase_id=?")->execute([$id]);
        // حدّث الرأس للحالة received
        $pdo->prepare("UPDATE purchases SET status='received' WHERE id=?")->execute([$id]);
        // هات الفرع
        $br = $pdo->prepare("SELECT branch_id FROM purchases WHERE id=?");
        $br->execute([$id]); $branch_id = (int)$br->fetchColumn();
        if ($branch_id) receive_and_move($pdo, $id, $branch_id);
        $success = '✔️ تم الاستلام بالكامل';

      } elseif ($action === 'add_payment') {
        $id = (int)($_POST['id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $method = clean_str($_POST['method'] ?? '', 40);
        $ref_no = clean_str($_POST['ref_no'] ?? '', 80);
        $notes  = clean_str($_POST['pay_notes'] ?? '', 255);
        if ($id<=0 || $amount<=0) throw new RuntimeException('مبلغ غير صالح.');
        $h = $pdo->prepare("SELECT supplier_id FROM purchases WHERE id=?");
        $h->execute([$id]); $sup = (int)$h->fetchColumn();
        if ($sup<=0) throw new RuntimeException('مورد غير معروف.');
        $pdo->prepare("INSERT INTO supplier_payments (supplier_id, purchase_id, amount, method, ref_no, notes)
                       VALUES (?,?,?,?,?,?)")->execute([$sup, $id, $amount, $method, $ref_no, $notes]);
        // حدّث paid في الرأس
        $sum = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM supplier_payments WHERE purchase_id=?");
        $sum->execute([$id]); $paid = (float)$sum->fetchColumn();
        $pdo->prepare("UPDATE purchases SET paid=? WHERE id=?")->execute([$paid, $id]);
        $success = '✔️ تم تسجيل الدفعة';

      } elseif ($action === 'soft_delete') {
        $id = (int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE purchases SET deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم أرشفة العملية';

      } elseif ($action === 'restore') {
        $id = (int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE purchases SET deleted_at=NULL WHERE id=?")->execute([$id]);
        $success = 'تم استعادة العملية';
      }

    } catch (Throwable $e) {
      $error = 'خطأ: '.htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
    }
  }
}

/* ==== فلاتر/عرض ==== */
$q            = trim($_GET['q'] ?? '');
$fSupplier    = ($_GET['supplier_id'] ?? '')!=='' ? (int)$_GET['supplier_id'] : null;
$fBranch      = ($_GET['branch_id'] ?? '')!=='' ? (int)$_GET['branch_id'] : null;
$fStatus      = $_GET['status'] ?? '';
$showArchived = isset($_GET['show_archived']);

$where = [];
$params = [];
$where[] = $showArchived ? "1=1" : "p.deleted_at IS NULL";
if ($q!==''){ $where[]="(p.invoice_no LIKE :q OR s.`{$SUPPLIER_NAME_COL}` LIKE :q)"; $params[':q']="%$q%"; }
if ($fSupplier){ $where[]="p.supplier_id=:sid"; $params[':sid']=$fSupplier; }
if ($fBranch){ $where[]="p.branch_id=:bid"; $params[':bid']=$fBranch; }
if ($fStatus!==''){ $where[]="p.status=:st"; $params[':st']=$fStatus; }
$whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';

$sql = "SELECT p.*, s.`{$SUPPLIER_NAME_COL}` AS SupplierName, b.BranchName,
        (p.total - p.paid) AS due
        FROM purchases p
        LEFT JOIN suppliers s ON s.id=p.supplier_id
        LEFT JOIN Branches b ON b.BranchID=p.branch_id
        $whereSql
        ORDER BY p.id DESC
        LIMIT 500";
$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

/* تحميل دفعات لكل عملية للعرض السريع */
$payMap = [];
if ($rows){
  $ids = array_column($rows,'id');
  $in  = implode(',', array_fill(0,count($ids),'?'));
  $ps  = $pdo->prepare("SELECT purchase_id, amount FROM supplier_payments WHERE purchase_id IN ($in)");
  $ps->execute($ids);
  foreach ($ps->fetchAll(PDO::FETCH_ASSOC) as $x){
    $pid = (int)$x['purchase_id'];
    $payMap[$pid] = ($payMap[$pid] ?? 0) + (float)$x['amount'];
  }
}
?>
<?php
// Endpoint JSON صغير لقراءة عملية واحدة (للاستخدام داخل نفس الصفحة)
if (isset($_GET['getPurchase'])) {
  header('Content-Type: application/json; charset=utf-8');
  $id = (int)$_GET['getPurchase'];
  [$h,$it] = load_purchase($pdo,$id);
  echo json_encode(['header'=>$h,'items'=>$it], JSON_UNESCAPED_UNICODE);
  exit;
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>المشتريات</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#ffffff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{
      background: linear-gradient(90deg, var(--p1), var(--p2), var(--p3), var(--p4));
      padding:14px 18px; color:#1f2a37; display:flex; align-items:center; gap:12px; justify-content:space-between;
      box-shadow:0 6px 24px rgba(0,0,0,.15);
    }
    .user-info{display:flex; align-items:center; gap:10px; background:rgba(255,255,255,.6); padding:6px 10px; border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}

    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .muted{color:var(--muted)}
    .table>:not(caption)>*>*{background:transparent}
    .num{min-width:90px;text-align:center}
    .item-row input{min-width:100px}
    .tot-box{background:rgba(0,0,0,.03); border-radius:12px; padding:10px}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-truck-field me-2"></i> المشتريات</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">

  <!-- شريط الفلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث بالفاتورة/اسم المورد" value="<?= htmlspecialchars($q) ?>">
        <select class="form-select" name="supplier_id" style="min-width:200px">
          <option value="">كل الموردين</option>
          <?php foreach ($suppliers as $sid=>$sname): ?>
            <option value="<?= (int)$sid ?>" <?= $fSupplier===$sid?'selected':''; ?>><?= htmlspecialchars($sname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="branch_id" style="min-width:200px">
          <option value="">كل الفروع</option>
          <?php foreach ($branches as $bid=>$bname): ?>
            <option value="<?= (int)$bid ?>" <?= $fBranch===$bid?'selected':''; ?>><?= htmlspecialchars($bname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="status" style="min-width:180px">
          <option value="">كل الحالات</option>
          <?php foreach (['draft'=>'مسودة','ordered'=>'أمر شراء','partial'=>'استلام جزئي','received'=>'تم الاستلام','cancelled'=>'ملغي'] as $k=>$v): ?>
            <option value="<?= $k ?>" <?= $fStatus===$k?'selected':''; ?>><?= $v ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $fSupplier || $fBranch || $fStatus || $showArchived): ?>
          <a href="purchases.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>

      <?php if ($canEdit): ?>
      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#purchaseModal" onclick="openCreate()">
        <i class="fa-solid fa-plus"></i> عملية جديدة
      </button>
      <?php endif; ?>
    </div>
  </div>

  <!-- جدول العمليات -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>التاريخ</th>
              <th>المورد</th>
              <th>الفرع</th>
              <th>فاتورة</th>
              <th>الحالة</th>
              <th>الإجمالي</th>
              <th>مدفوع</th>
              <th>متبقي</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="10" class="text-muted">لا توجد بيانات.</td></tr>
          <?php else: foreach ($rows as $r): ?>
            <tr data-pid="<?= (int)$r['id'] ?>">
              <td><?= (int)$r['id'] ?></td>
              <td><?= htmlspecialchars($r['order_date']) ?></td>
              <td><?= htmlspecialchars($r['SupplierName'] ?? '-') ?></td>
              <td><?= htmlspecialchars($r['BranchName'] ?? '-') ?></td>
              <td><?= htmlspecialchars($r['invoice_no'] ?? '—') ?></td>
              <td>
                <?php
                  $stt = [
                    'draft'=>'<span class="badge bg-secondary">مسودة</span>',
                    'ordered'=>'<span class="badge bg-info text-dark">أمر شراء</span>',
                    'partial'=>'<span class="badge bg-warning text-dark">استلام جزئي</span>',
                    'received'=>'<span class="badge bg-success">تم الاستلام</span>',
                    'cancelled'=>'<span class="badge bg-dark">ملغي</span>'
                  ];
                  echo $stt[$r['status']] ?? $r['status'];
                ?>
              </td>
              <td><?= money($r['total']) ?></td>
              <td><?= money($payMap[(int)$r['id']] ?? $r['paid']) ?></td>
              <td><?= money((float)$r['total'] - (float)($payMap[(int)$r['id']] ?? $r['paid'])) ?></td>
              <td>
                <div class="btn-group">
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <?php if ($r['status']!=='received' && $r['status']!=='cancelled'): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('تأكيد الاستلام الكامل؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="receive_all">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-box-open"></i></button>
                    </form>
                  <?php endif; ?>
                  <?php if ($r['deleted_at'] === null): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('أرشفة العملية؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="soft_delete">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                    </form>
                  <?php else: ?>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="restore">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                    </form>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>

    </div>
  </div>
</div>

<!-- Modal: Create/Edit Purchase -->
<div class="modal fade" id="purchaseModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-file-invoice-dollar me-2"></i> <span id="modalTitle">عملية جديدة</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>

      <form method="post" enctype="multipart/form-data" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save_purchase">
        <input type="hidden" name="id" id="f_id">

        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">المورد *</label>
            <select class="form-select" name="supplier_id" id="f_supplier" required>
              <option value="">— اختر —</option>
              <?php foreach ($suppliers as $sid=>$sname): ?>
                <option value="<?= (int)$sid ?>"><?= htmlspecialchars($sname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">الفرع</label>
            <select class="form-select" name="branch_id" id="f_branch">
              <option value="">— اختر —</option>
              <?php foreach ($branches as $bid=>$bname): ?>
                <option value="<?= (int)$bid ?>"><?= htmlspecialchars($bname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">الحالة</label>
            <select class="form-select" name="status" id="f_status">
              <option value="draft">مسودة</option>
              <option value="ordered">أمر شراء</option>
              <option value="partial">استلام جزئي</option>
              <option value="received">تم الاستلام</option>
              <option value="cancelled">ملغي</option>
            </select>
          </div>

          <div class="col-md-3">
            <label class="form-label">تاريخ الطلب</label>
            <input type="date" class="form-control" name="order_date" id="f_order_date" value="<?= date('Y-m-d') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">تاريخ متوقع</label>
            <input type="date" class="form-control" name="expected_date" id="f_expected_date">
          </div>
          <div class="col-md-3">
            <label class="form-label">رقم الفاتورة</label>
            <input type="text" class="form-control" name="invoice_no" id="f_invoice_no">
          </div>
          <div class="col-md-3">
            <label class="form-label">تاريخ الفاتورة</label>
            <input type="date" class="form-control" name="invoice_date" id="f_invoice_date">
          </div>

          <div class="col-md-2">
            <label class="form-label">العملة</label>
            <input type="text" class="form-control" name="currency_code" id="f_currency" value="EGP">
          </div>
          <div class="col-md-2">
            <label class="form-label">خصم عام</label>
            <div class="input-group">
              <select class="form-select" name="discount_type" id="f_discount_type" style="max-width:110px">
                <option value="fixed">قيمة</option>
                <option value="percent">%</option>
              </select>
              <input type="number" step="0.01" class="form-control" name="discount_value" id="f_discount_value" value="0">
            </div>
          </div>
          <div class="col-md-2">
            <label class="form-label">ضريبة %</label>
            <input type="number" step="0.01" class="form-control" name="tax_rate" id="f_tax_rate" value="0">
          </div>
          <div class="col-md-2">
            <label class="form-label">شحن</label>
            <input type="number" step="0.01" class="form-control" name="shipping" id="f_shipping" value="0">
          </div>
          <div class="col-md-2">
            <label class="form-label">رسوم</label>
            <input type="number" step="0.01" class="form-control" name="other_fees" id="f_other_fees" value="0">
          </div>
          <div class="col-md-2">
            <label class="form-label">مرفق (فاتورة)</label>
            <input type="file" class="form-control" name="attachment" accept="image/*,.pdf">
          </div>

          <div class="col-12">
            <label class="form-label">ملاحظات</label>
            <textarea class="form-control" name="notes" id="f_notes" rows="2"></textarea>
          </div>

          <!-- بنود -->
          <div class="col-12">
            <div class="table-responsive">
              <table class="table table-sm align-middle text-center" id="itemsTable">
                <thead class="table-light">
                  <tr>
                    <th style="min-width:240px">المتغيّر (SKU)</th>
                    <th>وصف</th>
                    <th class="num">كمية مطلوبة</th>
                    <th class="num">كمية مستلمة</th>
                    <th class="num">سعر شراء</th>
                    <th class="num">خصم سطر</th>
                    <th class="num">ضريبة %</th>
                    <th class="num">إجمالي سطر</th>
                    <th></th>
                  </tr>
                </thead>
                <tbody id="itemsBody"></tbody>
              </table>
            </div>
            <button type="button" class="btn btn-outline-primary" onclick="addRow()">➕ إضافة بند</button>
          </div>

          <!-- الإجماليات -->
          <div class="col-12">
            <div class="row g-3 justify-content-end">
              <div class="col-md-5">
                <div class="tot-box">
                  <div class="d-flex justify-content-between"><span>الإجمالي قبل الخصم/الضريبة:</span><strong id="t_subtotal">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>خصم عام:</span><strong id="t_hdrdisc">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>ضريبة عامة:</span><strong id="t_tax">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>شحن + رسوم:</span><strong id="t_extra">0.00</strong></div>
                  <hr class="my-2">
                  <div class="d-flex justify-content-between fs-5"><span>الإجمالي النهائي:</span><strong id="t_total">0.00</strong></div>
                </div>
              </div>
            </div>
          </div>

          <!-- دفعة سريعة (اختياري بعد الحفظ من جدول) نتركها من خارج المودال -->

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          </div>

        </div>
      </form>
    </div>
  </div>
</div>

<!-- Modal: Payment -->
<div class="modal fade" id="paymentModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-money-bill-wave me-2"></i> تسجيل دفعة</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <form method="post" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="add_payment">
        <input type="hidden" name="id" id="pay_purchase_id">
        <div class="mb-2">
          <label class="form-label">المبلغ</label>
          <input type="number" step="0.01" class="form-control" name="amount" required>
        </div>
        <div class="mb-2">
          <label class="form-label">الطريقة</label>
          <input type="text" class="form-control" name="method" placeholder="Cash/Bank...">
        </div>
        <div class="mb-2">
          <label class="form-label">مرجع</label>
          <input type="text" class="form-control" name="ref_no">
        </div>
        <div class="mb-2">
          <label class="form-label">ملاحظات</label>
          <input type="text" class="form-control" name="pay_notes">
        </div>
        <div class="text-center">
          <button class="btn btn-primary px-4">حفظ</button>
        </div>
      </form>
    </div>
  </div>
</div>

<datalist id="variantsList">
  <?php foreach ($variants as $v): ?>
    <option value="<?= (int)$v['id'] ?> | <?= htmlspecialchars($v['label'], ENT_QUOTES, 'UTF-8') ?>"></option>
  <?php endforeach; ?>
</datalist>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark mode
  const body = document.body;
  const darkSwitch = document.getElementById("darkSwitch");
  if (localStorage.getItem("darkMode")==="true"){ body.classList.add("dark"); darkSwitch.textContent="☀️"; }
  darkSwitch?.addEventListener("click", ()=> {
    body.classList.toggle("dark");
    const isDark = body.classList.contains("dark");
    localStorage.setItem("darkMode", isDark);
    darkSwitch.textContent = isDark ? "☀️" : "🌙";
  });

  const purchaseModal = new bootstrap.Modal(document.getElementById('purchaseModal'));
  const paymentModal  = new bootstrap.Modal(document.getElementById('paymentModal'));
  const itemsBody = document.getElementById('itemsBody');

  function openCreate(){
    document.getElementById('modalTitle').textContent='عملية جديدة';
    document.getElementById('f_id').value='';
    ['f_invoice_no','f_currency','f_notes','f_discount_value','f_tax_rate','f_shipping','f_other_fees'].forEach(id=>{
      const el=document.getElementById(id); if(el) el.value = (id==='f_currency'?'EGP':'0');
    });
    document.getElementById('f_discount_type').value='fixed';
    document.getElementById('f_status').value='draft';
    document.getElementById('f_order_date').value = new Date().toISOString().slice(0,10);
    document.getElementById('f_expected_date').value = '';
    document.getElementById('f_invoice_date').value = '';
    document.getElementById('f_supplier').value='';
    document.getElementById('f_branch').value='';
    itemsBody.innerHTML='';
    addRow();
    calcTotals();
  }

  // تحميل بيانات موجودة (fetch بنموذج بسيط)
  function openEdit(id){
    // نقرأ بـ fetch نسخة HTML للصفحة نفسها مع ?edit=ID (لنستخدم JSON داخل سكريبت؟ هنا هنستدعي API صغير)
    // للتبسيط: نفتح مودال فارغ ثم نطلب بيانات عبر mini endpoint داخلي باستخدام anchor tag مخفي؟
    // هنستخدم fetch لنفس الصفحة مع باراميتر getPurchase=ID وترجع JSON (نضيفه تحت في سكريبت PHP؟ لتبسيط سنعمل inline ajax هنا)

    fetch(location.href + (location.search ? '&':'?') + 'getPurchase=' + encodeURIComponent(id))
      .then(r=>r.json())
      .then(data=>{
        if(!data || !data.header){ alert('لم يتم العثور على البيانات'); return; }
        fillForm(data.header, data.items);
        purchaseModal.show();
      })
      .catch(()=> alert('تعذر التحميل'));
  }

  // نضيف endpoint بسيط يرجع JSON للشراء
</script>

<script>
  function fillForm(h, items){
    document.getElementById('modalTitle').textContent='تعديل عملية #' + (h.id||'');
    document.getElementById('f_id').value = h.id||'';
    document.getElementById('f_supplier').value = h.supplier_id||'';
    document.getElementById('f_branch').value = h.branch_id||'';
    document.getElementById('f_status').value = h.status||'draft';
    document.getElementById('f_order_date').value = h.order_date||'';
    document.getElementById('f_expected_date').value = h.expected_date||'';
    document.getElementById('f_invoice_no').value = h.invoice_no||'';
    document.getElementById('f_invoice_date').value = h.invoice_date||'';
    document.getElementById('f_currency').value = h.currency_code||'EGP';
    document.getElementById('f_discount_type').value = h.discount_type||'fixed';
    document.getElementById('f_discount_value').value = h.discount_value||0;
    document.getElementById('f_tax_rate').value = h.tax_rate||0;
    document.getElementById('f_shipping').value = h.shipping||0;
    document.getElementById('f_other_fees').value = h.other_fees||0;
    document.getElementById('f_notes').value = h.notes||'';
    itemsBody.innerHTML='';
    (items||[]).forEach(addRowFromItem);
    if ((items||[]).length===0) addRow();
    calcTotals();
  }

  function addRowFromItem(it){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)" value="${it.variant_id} | ${escapeHtml(it.product_name || '')} | ${escapeHtml(it.sku || '')}">
        <input type="hidden" name="item_variant_id[]" value="${it.variant_id}">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control" value="${escapeHtml(it.description||'')}"></td>
      <td><input type="number" name="item_ordered_qty[]" class="form-control text-center" value="${it.ordered_qty||0}" oninput="calcTotals()"></td>
      <td><input type="number" name="item_received_qty[]" class="form-control text-center" value="${it.received_qty||0}" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_unit_cost[]" class="form-control text-center" value="${it.unit_cost||0}" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="${it.line_discount||0}" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_tax[]" class="form-control text-center" value="${it.line_tax_rate||0}" oninput="calcTotals()"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
  }

  function addRow(){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)">
        <input type="hidden" name="item_variant_id[]">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control"></td>
      <td><input type="number" name="item_ordered_qty[]" class="form-control text-center" value="1" oninput="calcTotals()"></td>
      <td><input type="number" name="item_received_qty[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_unit_cost[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_tax[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
  }

  function parseVariant(inp){
    // يقبل إدخال مثل: "123 | اسم | SKU" ويستخرج ID
    const hidden = inp.parentElement.querySelector('input[type=hidden]');
    const val = (inp.value||'').trim();
    const id = parseInt(val.split('|')[0],10);
    if (!isNaN(id)) hidden.value = id;
    calcTotals();
  }

  function calcTotals(){
    let subtotal = 0;
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      const q  = parseFloat(tr.querySelector('[name="item_ordered_qty[]"]').value||0);
      const u  = parseFloat(tr.querySelector('[name="item_unit_cost[]"]').value||0);
      const d  = parseFloat(tr.querySelector('[name="item_line_discount[]"]').value||0);
      const t  = parseFloat(tr.querySelector('[name="item_line_tax[]"]').value||0);
      let line = (q*u) - d;
      let ltax = line * (t/100);
      let tot  = line + ltax;
      subtotal += tot;
      tr.querySelector('.line-total').textContent = tot.toFixed(2);
    });
    const dtype = document.getElementById('f_discount_type').value;
    const dval  = parseFloat(document.getElementById('f_discount_value').value||0);
    const taxR  = parseFloat(document.getElementById('f_tax_rate').value||0);
    const ship  = parseFloat(document.getElementById('f_shipping').value||0);
    const fees  = parseFloat(document.getElementById('f_other_fees').value||0);

    let hdrDisc = (dtype==='percent') ? (subtotal * (dval/100)) : dval;
    hdrDisc = Math.max(0, hdrDisc);
    const after = Math.max(0, subtotal - hdrDisc);
    const hdrTax = after * (taxR/100);
    const total = after + hdrTax + ship + fees;

    document.getElementById('t_subtotal').textContent = subtotal.toFixed(2);
    document.getElementById('t_hdrdisc').textContent  = hdrDisc.toFixed(2);
    document.getElementById('t_tax').textContent      = hdrTax.toFixed(2);
    document.getElementById('t_extra').textContent    = (ship+fees).toFixed(2);
    document.getElementById('t_total').textContent    = total.toFixed(2);
  }

  ['f_discount_type','f_discount_value','f_tax_rate','f_shipping','f_other_fees'].forEach(id=>{
    const el=document.getElementById(id); el && el.addEventListener('input', calcTotals);
  });

  function payFor(id){
    document.getElementById('pay_purchase_id').value = id;
    paymentModal.show();
  }

  function escapeHtml(s){ return (s||'').toString().replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
</script>
</body>
</html>
