<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ===== Helpers ===== */
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function money($n){ return number_format((float)$n, 2); }
function clean_str(?string $s, int $max=255): ?string{ $s=trim((string)$s); if($s==='') return null; return mb_substr($s,0,$max); }
function hasColumn(PDO $pdo, string $table, string $col): bool{
  $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name=? AND column_name=?");
  $st->execute([$table,$col]); return (int)$st->fetchColumn()>0;
}
function tableExists(PDO $pdo, string $table): bool{
  $st=$pdo->prepare("SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name=?");
  $st->execute([$table]); return (int)$st->fetchColumn()>0;
}
function detectCols(PDO $pdo, string $table, array $idCands, array $nameCands): array {
  $idCol = $idCands[0]; $nmCol = $nameCands[0];
  foreach ($idCands as $c)  if (hasColumn($pdo,$table,$c)) { $idCol=$c; break; }
  foreach ($nameCands as $c)if (hasColumn($pdo,$table,$c)) { $nmCol=$c; break; }
  return [$idCol,$nmCol];
}

/* ===== CSRF ===== */
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf_token'];

/* ===== إنشاء الجداول (إن لم توجد) ===== */
$pdo->exec("
CREATE TABLE IF NOT EXISTS sales_returns (
  id INT AUTO_INCREMENT PRIMARY KEY,
  branch_id INT NOT NULL,
  customer_id INT NULL,
  sale_id INT NULL,
  status ENUM('draft','approved','partial','restocked','cancelled') NOT NULL DEFAULT 'draft',
  return_date DATE NOT NULL,
  currency_code VARCHAR(10) NOT NULL DEFAULT 'EGP',
  subtotal DECIMAL(12,2) NOT NULL DEFAULT 0.00,         -- شامل الضريبة
  discount_type ENUM('fixed','percent') NOT NULL DEFAULT 'fixed',
  discount_value DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  tax_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,          -- نسبة مرجعية للاستخراج فقط
  tax_amount_extracted DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  rounding_adjust DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  total DECIMAL(12,2) NOT NULL DEFAULT 0.00,            -- قيمة الرصيد/الاسترداد شامل
  notes TEXT NULL,
  deleted_at DATETIME NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS sales_return_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  return_id INT NOT NULL,
  variant_id INT NOT NULL,
  description VARCHAR(255) NULL,
  qty INT NOT NULL DEFAULT 1,                -- المطلوب إرجاعه
  processed_qty INT NOT NULL DEFAULT 0,      -- الذي تم إرجاعه فعلاً للمخزون
  unit_price_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,  -- سعر بيع شامل
  line_discount DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  tax_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  line_total_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  line_tax_extracted DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  CONSTRAINT fk_sr_item_return FOREIGN KEY (return_id) REFERENCES sales_returns(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS customer_refunds (
  id INT AUTO_INCREMENT PRIMARY KEY,
  customer_id INT NULL,
  return_id INT NOT NULL,
  amount DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  method VARCHAR(40) NULL,
  ref_no VARCHAR(80) NULL,
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

/* ===== اكتشاف جداول الفروع/العملاء ===== */
$branchesTable = tableExists($pdo,'Branches') ? 'Branches' : (tableExists($pdo,'branches') ? 'branches' : 'branches');
[$BR_ID_COL,$BR_NAME_COL] = detectCols($pdo,$branchesTable,['id','branch_id','BranchID'],['name','branch_name','BranchName']);
$branches = [];
try {
  $branches = $pdo->query("SELECT `$BR_ID_COL` AS id, `$BR_NAME_COL` AS name FROM `$branchesTable` ORDER BY `$BR_NAME_COL`")->fetchAll(PDO::FETCH_KEY_PAIR);
} catch(Throwable $e){ $branches=[]; }

$customersTable = tableExists($pdo,'customers') ? 'customers' : 'customers';
[$CUS_ID_COL,$CUS_NAME_COL] = detectCols($pdo,$customersTable,['id','CustomerID'],['name','CustomerName','full_name']);
$customers=[];
try{
  $customers = $pdo->query("SELECT `$CUS_ID_COL` AS id, `$CUS_NAME_COL` AS name FROM `$customersTable` ORDER BY `$CUS_NAME_COL`")->fetchAll(PDO::FETCH_KEY_PAIR);
}catch(Throwable $e){ $customers=[]; }

/* إعدادات الضريبة (نسبة مرجعية) */
$settings = ['vat_rate'=>14.00];
try{
  $row = $pdo->query("SELECT * FROM settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);
  if ($row){
    foreach (['vat_rate','VAT','vat','tax_rate','TaxRate'] as $k){
      if (isset($row[$k])) { $settings['vat_rate'] = (float)$row[$k]; break; }
    }
  }
}catch(Throwable $e){}

/* متغيرات للبحث السريع */
$variants = $pdo->query("
  SELECT v.id, v.sku,
         CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-'),' | ',v.sku) AS label
  FROM product_variants v
  JOIN products p ON p.id=v.product_id
  WHERE v.deleted_at IS NULL
  ORDER BY p.name_ar, v.id
  LIMIT 2000
")->fetchAll(PDO::FETCH_ASSOC);

/* ===== حساب الإجماليات (أسعار شاملة الضريبة) ===== */
function calc_totals(array $items, string $discount_type, float $discount_value, float $header_tax_rate): array{
  $subtotal_incl = 0.0;
  $tax_extracted = 0.0;
  foreach ($items as $it){
    $qty = max(0,(int)$it['qty']);
    $unit = max(0.0,(float)$it['unit_price_incl']);
    $ld   = max(0.0,(float)$it['line_discount']);
    $line_incl = max(0.0, ($qty*$unit) - $ld);
    $subtotal_incl += $line_incl;

    $t = isset($it['tax_rate']) ? (float)$it['tax_rate'] : $header_tax_rate;
    $net = $t>0 ? ($line_incl/(1+$t/100)) : $line_incl;
    $tax_extracted += ($line_incl - $net);
  }
  $hdrDisc = ($discount_type==='percent') ? ($subtotal_incl*($discount_value/100)) : $discount_value;
  $hdrDisc = max(0.0,$hdrDisc);
  $after = max(0.0, $subtotal_incl - $hdrDisc);

  // تقريب لأقرب 0.05 (اختياري، شائع في النظام)
  $rounded = round($after/0.05)*0.05;
  $adj = $rounded - $after;

  return [$subtotal_incl,$tax_extracted,$hdrDisc,$adj,$rounded];
}

/* ===== حركة المخزون عند "إرجاع للمخزون" ===== */
function last_purchase_cost(PDO $pdo, int $variantId): float {
  try {
    $st=$pdo->prepare("SELECT unit_cost FROM purchase_items WHERE variant_id=? ORDER BY id DESC LIMIT 1");
    $st->execute([$variantId]); $x=$st->fetchColumn();
    return $x!==false ? (float)$x : 0.0;
  } catch(Throwable $e){ return 0.0; }
}
function restock_and_move(PDO $pdo, int $return_id, int $branch_id){
  $stmt = $pdo->prepare("SELECT id, variant_id, qty, processed_qty, unit_price_incl FROM sales_return_items WHERE return_id=? ORDER BY id");
  $stmt->execute([$return_id]);
  $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
  foreach ($items as $it){
    $prev = (int)($GLOBALS['_prev_processed'][$it['id']] ?? 0);
    $delta = (int)$it['processed_qty'] - $prev;
    if ($delta > 0 && $branch_id){
      // زيادة المخزون
      $up = $pdo->prepare("
        INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
        VALUES (?,?,?,0)
        ON DUPLICATE KEY UPDATE quantity = quantity + VALUES(quantity)
      ");
      $up->execute([(int)$it['variant_id'],$branch_id,$delta]);

      // دفتر حركة
      $uc = last_purchase_cost($pdo,(int)$it['variant_id']);
      $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                     VALUES (?,?,?,?, 'sale_return', ?)")->execute([
        (int)$it['variant_id'], $branch_id, +$delta, $uc, $return_id
      ]);
    }
  }
}

/* للاحتفاظ بالقيم السابقة قبل التحديث */
$_prev_processed = [];

/* ===== JSON: تحميل مرتجع واحد ===== */
function load_return(PDO $pdo, int $id){
  $h = $pdo->prepare("SELECT * FROM sales_returns WHERE id=?");
  $h->execute([$id]); $header = $h->fetch(PDO::FETCH_ASSOC);
  if (!$header) return [null,[]];
  $i = $pdo->prepare("
    SELECT ri.*, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM sales_return_items ri
    JOIN product_variants v ON v.id=ri.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE ri.return_id=?
    ORDER BY ri.id
  ");
  $i->execute([$id]);
  return [$header, $i->fetchAll(PDO::FETCH_ASSOC)];
}
if (isset($_GET['getReturn'])) {
  header('Content-Type: application/json; charset=utf-8');
  $id = (int)$_GET['getReturn'];
  [$h,$it] = load_return($pdo,$id);
  echo json_encode(['header'=>$h,'items'=>$it], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== POST Actions ===== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF).';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action==='save_return') {
        $id           = ($_POST['id'] ?? '')!=='' ? (int)$_POST['id'] : null;
        $branch_id    = (int)($_POST['branch_id'] ?? 0);
        $customer_id  = ($_POST['customer_id'] ?? '')!=='' ? (int)$_POST['customer_id'] : null;
        $sale_id      = ($_POST['sale_id'] ?? '')!=='' ? (int)$_POST['sale_id'] : null;
        $status       = $_POST['status'] ?? 'draft';
        $return_date  = $_POST['return_date'] ?: date('Y-m-d');
        $currency     = clean_str($_POST['currency_code'] ?? 'EGP', 10);
        $discount_type= $_POST['discount_type'] ?? 'fixed';
        $discount_val = (float)($_POST['discount_value'] ?? 0);
        $tax_rate     = (float)($_POST['tax_rate'] ?? (float)$settings['vat_rate']);
        $notes        = clean_str($_POST['notes'] ?? '', 10000);

        if ($branch_id<=0) throw new RuntimeException('اختر فرعًا.');

        // بنود
        $var_ids = array_map('intval', $_POST['item_variant_id'] ?? []);
        $desc_a  = $_POST['item_desc'] ?? [];
        $qty_a   = array_map('intval', $_POST['item_qty'] ?? []);
        $proc_a  = array_map('intval', $_POST['item_processed_qty'] ?? []);
        $price_a = array_map('floatval', $_POST['item_unit_price'] ?? []);
        $disc_a  = array_map('floatval', $_POST['item_line_discount'] ?? []);
        $tax_a   = array_map('floatval', $_POST['item_line_tax'] ?? []);

        $items=[];
        for($i=0;$i<count($var_ids);$i++){
          if ($var_ids[$i] <= 0) continue;
          $qty  = max(0, (int)($qty_a[$i] ?? 0));
          $proc = max(0, (int)($proc_a[$i] ?? 0));
          if ($proc > $qty) $proc=$qty;
          $unit = max(0.0,(float)($price_a[$i] ?? 0));
          $ld   = max(0.0,(float)($disc_a[$i] ?? 0));
          $lt   = max(0.0,(float)($tax_a[$i] ?? $tax_rate));
          $desc = clean_str($desc_a[$i] ?? '', 255);
          if ($qty===0 && $proc===0) continue;
          $items[] = [
            'variant_id'=>$var_ids[$i], 'description'=>$desc,
            'qty'=>$qty, 'processed_qty'=>$proc,
            'unit_price_incl'=>$unit, 'line_discount'=>$ld, 'line_tax_rate'=>$lt
          ];
        }
        if (!$items) throw new RuntimeException('أضف بندًا واحدًا على الأقل.');

        // إجماليات
        [$subtotal,$taxExt,$hdrDisc,$roundAdj,$total] = calc_totals($items, $discount_type, $discount_val, $tax_rate);

        if ($id){
          // خزّن processed السابق
          $pr=$pdo->prepare("SELECT id, processed_qty FROM sales_return_items WHERE return_id=?");
          $pr->execute([$id]); foreach($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_processed[(int)$r['id']] = (int)$r['processed_qty']; }

          $sql="UPDATE sales_returns SET branch_id=:b, customer_id=:c, sale_id=:sid, status=:st, return_date=:dt,
                currency_code=:cc, subtotal=:sub, discount_type=:dtype, discount_value=:dval, tax_rate=:tr,
                tax_amount_extracted=:tax, rounding_adjust=:adj, total=:tot, notes=:notes
                WHERE id=:id AND deleted_at IS NULL";
          $st=$pdo->prepare($sql);
          $st->execute([
            ':b'=>$branch_id, ':c'=>$customer_id, ':sid'=>$sale_id, ':st'=>$status, ':dt'=>$return_date,
            ':cc'=>$currency, ':sub'=>$subtotal, ':dtype'=>$discount_type, ':dval'=>$discount_val, ':tr'=>$tax_rate,
            ':tax'=>$taxExt, ':adj'=>$roundAdj, ':tot'=>$total, ':notes'=>$notes, ':id'=>$id
          ]);

          $pdo->prepare("DELETE FROM sales_return_items WHERE return_id=?")->execute([$id]);
          $ins=$pdo->prepare("INSERT INTO sales_return_items
            (return_id, variant_id, description, qty, processed_qty, unit_price_incl, line_discount, line_tax_rate, line_total_incl, line_tax_extracted)
            VALUES (?,?,?,?,?,?,?,?,?,?)");
          foreach ($items as $it){
            $line_incl = max(0.0, ($it['qty']*$it['unit_price_incl']) - $it['line_discount']);
            $net = $it['line_tax_rate']>0 ? ($line_incl/(1+$it['line_tax_rate']/100)) : $line_incl;
            $tax = $line_incl - $net;
            $ins->execute([$id,$it['variant_id'],$it['description'],$it['qty'],$it['processed_qty'],$it['unit_price_incl'],$it['line_discount'],$it['line_tax_rate'],$line_incl,$tax]);
          }

          if (in_array($status,['partial','restocked'], true) && $branch_id){
            restock_and_move($pdo,$id,$branch_id);
          }

          $success = '✔️ تم تحديث مرتجع المبيعات';

        } else {
          $st=$pdo->prepare("INSERT INTO sales_returns
            (branch_id, customer_id, sale_id, status, return_date, currency_code, subtotal, discount_type, discount_value, tax_rate, tax_amount_extracted, rounding_adjust, total, notes)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
          $st->execute([$branch_id,$customer_id,$sale_id,$status,$return_date,$currency,$subtotal,$discount_type,$discount_val,$tax_rate,$taxExt,$roundAdj,$total,$notes]);
          $rid=(int)$pdo->lastInsertId();

          $ins=$pdo->prepare("INSERT INTO sales_return_items
            (return_id, variant_id, description, qty, processed_qty, unit_price_incl, line_discount, line_tax_rate, line_total_incl, line_tax_extracted)
            VALUES (?,?,?,?,?,?,?,?,?,?)");
          foreach ($items as $it){
            $line_incl = max(0.0, ($it['qty']*$it['unit_price_incl']) - $it['line_discount']);
            $net = $it['line_tax_rate']>0 ? ($line_incl/(1+$it['line_tax_rate']/100)) : $line_incl;
            $tax = $line_incl - $net;
            $ins->execute([$rid,$it['variant_id'],$it['description'],$it['qty'],$it['processed_qty'],$it['unit_price_incl'],$it['line_discount'],$it['line_tax_rate'],$line_incl,$tax]);
          }

          if (in_array($status,['partial','restocked'], true) && $branch_id){
            $_prev_processed = [];
            restock_and_move($pdo,$rid,$branch_id);
          }

          $success = '✔️ تم إنشاء مرتجع المبيعات';
        }

      } elseif ($action==='execute_all') {
        $id = (int)($_POST['id'] ?? 0);
        if ($id<=0) throw new RuntimeException('معرّف غير صالح.');
        // السابق
        $pr=$pdo->prepare("SELECT id, processed_qty FROM sales_return_items WHERE return_id=?");
        $pr->execute([$id]); foreach($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_processed[(int)$r['id']] = (int)$r['processed_qty']; }
        // تنفيذ كامل
        $pdo->prepare("UPDATE sales_return_items SET processed_qty = qty WHERE return_id=?")->execute([$id]);
        $pdo->prepare("UPDATE sales_returns SET status='restocked' WHERE id=?")->execute([$id]);
        // فرع العملية
        $br=$pdo->prepare("SELECT branch_id FROM sales_returns WHERE id=?"); $br->execute([$id]); $branch_id=(int)$br->fetchColumn();
        if ($branch_id) restock_and_move($pdo,$id,$branch_id);
        $success = '✔️ تم إرجاع كل الكميات للمخزون';

      } elseif ($action==='add_refund') {
        $rid = (int)($_POST['id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $method = clean_str($_POST['method'] ?? '', 40);
        $ref_no = clean_str($_POST['ref_no'] ?? '', 80);
        $notes  = clean_str($_POST['refund_notes'] ?? '', 255);
        if ($rid<=0 || $amount<=0) throw new RuntimeException('مبلغ غير صالح.');
        $st = $pdo->prepare("SELECT customer_id FROM sales_returns WHERE id=?");
        $st->execute([$rid]); $cus = $st->fetchColumn();
        $pdo->prepare("INSERT INTO customer_refunds (customer_id, return_id, amount, method, ref_no, notes) VALUES (?,?,?,?,?,?)")
            ->execute([$cus?:null,$rid,$amount,$method,$ref_no,$notes]);
        $success = '✔️ تم تسجيل استرداد للعميل';

      } elseif ($action==='soft_delete') {
        $id=(int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE sales_returns SET deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم أرشفة المرتجع';

      } elseif ($action==='restore') {
        $id=(int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE sales_returns SET deleted_at=NULL WHERE id=?")->execute([$id]);
        $success = 'تم استعادة المرتجع';
      }

    } catch(Throwable $e){
      $error = 'خطأ: '.esc($e->getMessage());
    }
  }
}

/* ===== فلاتر العرض ===== */
$q            = trim($_GET['q'] ?? '');
$fCustomer    = ($_GET['customer_id'] ?? '')!=='' ? (int)$_GET['customer_id'] : null;
$fBranch      = ($_GET['branch_id'] ?? '')!=='' ? (int)$_GET['branch_id'] : null;
$fStatus      = $_GET['status'] ?? '';
$showArchived = isset($_GET['show_archived']);

$where=[]; $params=[];
$where[] = $showArchived ? "1=1" : "r.deleted_at IS NULL";
if ($q!==''){ $where[]="(r.id LIKE :q OR c.`$CUS_NAME_COL` LIKE :q)"; $params[':q']="%$q%"; }
if ($fCustomer){ $where[]="r.customer_id=:cid"; $params[':cid']=$fCustomer; }
if ($fBranch){ $where[]="r.branch_id=:bid"; $params[':bid']=$fBranch; }
if ($fStatus!==''){ $where[]="r.status=:st"; $params[':st']=$fStatus; }
$whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';

$sql = "SELECT r.*, b.`$BR_NAME_COL` AS BranchName, c.`$CUS_NAME_COL` AS CustomerName
        FROM sales_returns r
        LEFT JOIN `$branchesTable` b ON b.`$BR_ID_COL`=r.branch_id
        LEFT JOIN `$customersTable` c ON c.`$CUS_ID_COL`=r.customer_id
        $whereSql
        ORDER BY r.id DESC
        LIMIT 500";
$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

/* تجميع استردادات نقدية لكل مرتجع */
$refundMap = [];
if ($rows){
  $ids = array_column($rows,'id');
  $in  = implode(',', array_fill(0,count($ids),'?'));
  $ps  = $pdo->prepare("SELECT return_id, SUM(amount) as sum_amount FROM customer_refunds WHERE return_id IN ($in) GROUP BY return_id");
  $ps->execute($ids);
  foreach ($ps->fetchAll(PDO::FETCH_ASSOC) as $x){
    $refundMap[(int)$x['return_id']] = (float)$x['sum_amount'];
  }
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>مرتجع مبيعات</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#fff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{ background:linear-gradient(90deg,var(--p1),var(--p2),var(--p3),var(--p4)); padding:14px 18px; display:flex; align-items:center; justify-content:space-between; box-shadow:0 6px 24px rgba(0,0,0,.15); }
    .user-info{display:flex;align-items:center;gap:10px;background:rgba(255,255,255,.6);padding:6px 10px;border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}
    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .muted{color:var(--muted)}
    .num{min-width:90px;text-align:center}
    .table>:not(caption)>*>*{background:transparent}
    .tot-box{background:rgba(0,0,0,.03); border-radius:12px; padding:10px}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= esc($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= esc($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-rotate-left me-2"></i> مرتجع مبيعات</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">
  <!-- فلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث برقم المرتجع/اسم العميل" value="<?= esc($q) ?>">
        <select class="form-select" name="customer_id" style="min-width:200px">
          <option value="">كل العملاء</option>
          <?php foreach ($customers as $cid=>$cname): ?>
            <option value="<?= (int)$cid ?>" <?= $fCustomer===$cid?'selected':''; ?>><?= esc($cname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="branch_id" style="min-width:200px">
          <option value="">كل الفروع</option>
          <?php foreach ($branches as $bid=>$bname): ?>
            <option value="<?= (int)$bid ?>" <?= $fBranch===$bid?'selected':''; ?>><?= esc($bname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="status" style="min-width:200px">
          <option value="">كل الحالات</option>
          <?php foreach (['draft'=>'مسودة','approved'=>'مقبول','partial'=>'تنفيذ جزئي','restocked'=>'أعيد للمخزون','cancelled'=>'ملغي'] as $k=>$v): ?>
            <option value="<?= $k ?>" <?= $fStatus===$k?'selected':''; ?>><?= $v ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $fCustomer || $fBranch || $fStatus || $showArchived): ?>
          <a href="sales_returns.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>

      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#returnModal" onclick="openCreate()">
        <i class="fa-solid fa-plus"></i> مرتجع جديد
      </button>
    </div>
  </div>

  <!-- جدول -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>التاريخ</th>
              <th>العميل</th>
              <th>الفرع</th>
              <th>الحالة</th>
              <th>الإجمالي (رصيد)</th>
              <th>مسترد</th>
              <th>متبقي</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="9" class="text-muted">لا توجد بيانات.</td></tr>
          <?php else: foreach ($rows as $r):
              $refunded = (float)($refundMap[(int)$r['id']] ?? 0);
              $due = (float)$r['total'] - $refunded;
          ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>
              <td><?= esc($r['return_date']) ?></td>
              <td><?= esc($r['CustomerName'] ?? 'عميل نقدي') ?></td>
              <td><?= esc($r['BranchName'] ?? '-') ?></td>
              <td>
                <?php
                  $badges = [
                    'draft'=>'<span class="badge bg-secondary">مسودة</span>',
                    'approved'=>'<span class="badge bg-info text-dark">مقبول</span>',
                    'partial'=>'<span class="badge bg-warning text-dark">تنفيذ جزئي</span>',
                    'restocked'=>'<span class="badge bg-success">أعيد للمخزون</span>',
                    'cancelled'=>'<span class="badge bg-dark">ملغي</span>'
                  ];
                  echo $badges[$r['status']] ?? esc($r['status']);
                ?>
              </td>
              <td><?= money($r['total']) ?></td>
              <td><?= money($refunded) ?></td>
              <td><?= money($due) ?></td>
              <td>
                <div class="btn-group">
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <?php if ($r['status']!=='restocked' && $r['status']!=='cancelled'): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('تنفيذ المرتجع بالكامل؟ سيُضاف للمخزون.')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="execute_all">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-dolly"></i></button>
                    </form>
                  <?php endif; ?>
                  <button class="btn btn-sm btn-outline-primary" onclick="openRefund(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-receipt"></i></button>
                  <?php if ($r['deleted_at'] === null): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('أرشفة المرتجع؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="soft_delete">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                    </form>
                  <?php else: ?>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="restore">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                    </form>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<!-- Modal: Create/Edit Sales Return -->
<div class="modal fade" id="returnModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-box-open me-2"></i> <span id="modalTitle">مرتجع جديد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>

      <form method="post" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save_return">
        <input type="hidden" name="id" id="f_id">

        <div class="row g-3">
          <div class="col-md-3">
            <label class="form-label">الفرع *</label>
            <select class="form-select" name="branch_id" id="f_branch" required>
              <option value="">— اختر —</option>
              <?php foreach ($branches as $bid=>$bname): ?>
                <option value="<?= (int)$bid ?>"><?= esc($bname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">العميل</label>
            <select class="form-select" name="customer_id" id="f_customer">
              <option value="">عميل نقدي</option>
              <?php foreach ($customers as $cid=>$cname): ?>
                <option value="<?= (int)$cid ?>"><?= esc($cname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">الحالة</label>
            <select class="form-select" name="status" id="f_status">
              <option value="draft">مسودة</option>
              <option value="approved">مقبول</option>
              <option value="partial">تنفيذ جزئي</option>
              <option value="restocked">أعيد للمخزون</option>
              <option value="cancelled">ملغي</option>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">تاريخ المرتجع</label>
            <input type="date" class="form-control" name="return_date" id="f_return_date" value="<?= date('Y-m-d') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">فاتورة بيع (اختياري)</label>
            <input type="number" class="form-control" name="sale_id" id="f_sale_id" placeholder="رقم الفاتورة">
          </div>
          <div class="col-md-2">
            <label class="form-label">العملة</label>
            <input type="text" class="form-control" name="currency_code" id="f_currency" value="EGP">
          </div>
          <div class="col-md-2">
            <label class="form-label">ضريبة % (شامل)</label>
            <input type="number" step="0.01" class="form-control" name="tax_rate" id="f_tax_rate" value="<?= (float)$settings['vat_rate'] ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">خصم عام</label>
            <div class="input-group">
              <select class="form-select" name="discount_type" id="f_discount_type" style="max-width:110px">
                <option value="fixed">قيمة</option>
                <option value="percent">%</option>
              </select>
              <input type="number" step="0.01" class="form-control" name="discount_value" id="f_discount_value" value="0">
            </div>
          </div>

          <div class="col-12">
            <label class="form-label">ملاحظات</label>
            <textarea class="form-control" name="notes" id="f_notes" rows="2"></textarea>
          </div>

          <!-- بنود -->
          <div class="col-12">
            <div class="table-responsive">
              <table class="table table-sm align-middle text-center">
                <thead class="table-light">
                  <tr>
                    <th style="min-width:240px">المتغيّر (SKU)</th>
                    <th>وصف</th>
                    <th class="num">كمية مطلوبة</th>
                    <th class="num">كمية منفّذة</th>
                    <th class="num">سعر شامل</th>
                    <th class="num">خصم سطر</th>
                    <th class="num">ضريبة %</th>
                    <th class="num">إجمالي سطر</th>
                    <th></th>
                  </tr>
                </thead>
                <tbody id="itemsBody"></tbody>
              </table>
            </div>
            <button type="button" class="btn btn-outline-primary" onclick="addRow()">➕ إضافة بند</button>
          </div>

          <!-- الإجماليات -->
          <div class="col-12">
            <div class="row g-3 justify-content-end">
              <div class="col-md-5">
                <div class="tot-box">
                  <div class="d-flex justify-content-between"><span>المجموع (شامل):</span><strong id="t_subtotal">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>خصم عام:</span><strong id="t_hdrdisc">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>تقريب 0.05:</span><strong id="t_round">0.00</strong></div>
                  <hr class="my-2">
                  <div class="d-flex justify-content-between fs-5"><span>الإجمالي (رصيد دائن):</span><strong id="t_total">0.00</strong></div>
                </div>
              </div>
            </div>
          </div>

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          </div>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Modal: Refund -->
<div class="modal fade" id="refundModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-money-bill-transfer me-2"></i> تسجيل استرداد للعميل</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="add_refund">
      <input type="hidden" name="id" id="refund_return_id">
      <div class="mb-2">
        <label class="form-label">المبلغ</label>
        <input type="number" step="0.01" class="form-control" name="amount" required>
      </div>
      <div class="mb-2">
        <label class="form-label">الطريقة</label>
        <input type="text" class="form-control" name="method" placeholder="Cash/Card/Transfer">
      </div>
      <div class="mb-2">
        <label class="form-label">مرجع</label>
        <input type="text" class="form-control" name="ref_no">
      </div>
      <div class="mb-2">
        <label class="form-label">ملاحظات</label>
        <input type="text" class="form-control" name="refund_notes">
      </div>
      <div class="text-center"><button class="btn btn-primary px-4">حفظ</button></div>
    </form>
  </div></div>
</div>

<datalist id="variantsList">
  <?php foreach ($variants as $v): ?>
    <option value="<?= (int)$v['id'] ?> | <?= esc($v['label']) ?>"></option>
  <?php endforeach; ?>
</datalist>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark
  const body=document.body, darkSwitch=document.getElementById('darkSwitch');
  if (localStorage.getItem('darkMode')==='true'){ body.classList.add('dark'); darkSwitch.textContent='☀️'; }
  darkSwitch?.addEventListener('click', ()=>{ body.classList.toggle('dark'); const d=body.classList.contains('dark'); localStorage.setItem('darkMode', d); darkSwitch.textContent = d?'☀️':'🌙'; });

  const returnModal = new bootstrap.Modal(document.getElementById('returnModal'));
  const refundModal = new bootstrap.Modal(document.getElementById('refundModal'));
  const itemsBody = document.getElementById('itemsBody');

  function openCreate(){
    document.getElementById('modalTitle').textContent='مرتجع جديد';
    ['f_id','f_customer','f_sale_id'].forEach(id=>{ const el=document.getElementById(id); if(el) el.value=''; });
    ['f_notes','f_discount_value'].forEach(id=>{ const el=document.getElementById(id); if(el) el.value = (id==='f_discount_value'?'0':''); });
    document.getElementById('f_status').value='draft';
    document.getElementById('f_discount_type').value='fixed';
    document.getElementById('f_return_date').value = new Date().toISOString().slice(0,10);
    document.getElementById('f_currency').value='EGP';
    document.getElementById('f_tax_rate').value='<?= (float)$settings['vat_rate'] ?>';
    itemsBody.innerHTML=''; addRow(); calcTotals();
  }

  function openEdit(id){
    fetch(location.href + (location.search ? '&':'?') + 'getReturn=' + encodeURIComponent(id))
      .then(r=>r.json())
      .then(data=>{
        if(!data || !data.header){ alert('لم يتم العثور على البيانات'); return; }
        fillForm(data.header, data.items);
        returnModal.show();
      })
      .catch(()=> alert('تعذر التحميل'));
  }

  function fillForm(h, items){
    document.getElementById('modalTitle').textContent='تعديل مرتجع #' + (h.id||'');
    document.getElementById('f_id').value = h.id||'';
    document.getElementById('f_branch').value = h.branch_id||'';
    document.getElementById('f_customer').value = h.customer_id||'';
    document.getElementById('f_sale_id').value = h.sale_id||'';
    document.getElementById('f_status').value = h.status||'draft';
    document.getElementById('f_return_date').value = h.return_date||'';
    document.getElementById('f_currency').value = h.currency_code||'EGP';
    document.getElementById('f_discount_type').value = h.discount_type||'fixed';
    document.getElementById('f_discount_value').value = h.discount_value||0;
    document.getElementById('f_tax_rate').value = h.tax_rate||0;
    document.getElementById('f_notes').value = h.notes||'';
    itemsBody.innerHTML='';
    (items||[]).forEach(addRowFromItem);
    if ((items||[]).length===0) addRow();
    calcTotals();
  }

  function addRowFromItem(it){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)" value="${it.variant_id} | ${(it.product_name||'').replaceAll('"','&quot;')} | ${(it.sku||'')}">
        <input type="hidden" name="item_variant_id[]" value="${it.variant_id}">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control" value="${(it.description||'').replaceAll('"','&quot;')}"></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="${it.qty||1}" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="number" name="item_processed_qty[]" class="form-control text-center" value="${it.processed_qty||0}" oninput="clampProcessed(this.closest('tr'));"></td>
      <td><input type="number" step="0.01" name="item_unit_price[]" class="form-control text-center" value="${it.unit_price_incl||0}" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="${it.line_discount||0}" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_tax[]" class="form-control text-center" value="${it.tax_rate ?? (document.getElementById('f_tax_rate').value||0)}" oninput="calcTotals()"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
    calcLine(tr);
  }

  function addRow(){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)">
        <input type="hidden" name="item_variant_id[]">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control"></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="1" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="number" name="item_processed_qty[]" class="form-control text-center" value="0" oninput="clampProcessed(this.closest('tr'));"></td>
      <td><input type="number" step="0.01" name="item_unit_price[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_tax[]" class="form-control text-center" value="<?= (float)$settings['vat_rate'] ?>" oninput="calcTotals()"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
    calcLine(tr);
  }

  function clampProcessed(tr){
    const qty = parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
    const proc = tr.querySelector('[name="item_processed_qty[]"]');
    let v = parseFloat(proc.value||0); if (v>qty) v=qty; if (v<0) v=0; proc.value = v;
    calcLine(tr);
  }

  function parseVariant(inp){
    const hidden = inp.parentElement.querySelector('input[type=hidden]');
    const val = (inp.value||'').trim();
    const id = parseInt(val.split('|')[0],10);
    if (!isNaN(id)) hidden.value = id;
    calcLine(inp.closest('tr'));
  }

  function calcLine(tr){
    const q  = parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
    const u  = parseFloat(tr.querySelector('[name="item_unit_price[]"]').value||0);
    const d  = parseFloat(tr.querySelector('[name="item_line_discount[]"]').value||0);
    const t  = parseFloat(tr.querySelector('[name="item_line_tax[]"]').value||0);
    let line = Math.max(0, (q*u) - d);
    // الأسعار شاملة؛ لا نضيف الضريبة على الإجمالي، لكن نستخرجها للتقارير (جانب السيرفر)
    tr.querySelector('.line-total').textContent = line.toFixed(2);
    calcTotals();
  }

  function calcTotals(){
    let subtotal=0;
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      subtotal += parseFloat(tr.querySelector('.line-total').textContent||0);
    });
    const dtype = document.getElementById('f_discount_type').value;
    const dval  = parseFloat(document.getElementById('f_discount_value').value||0);
    let hdrDisc = (dtype==='percent') ? (subtotal*(dval/100)) : dval; hdrDisc=Math.max(0,hdrDisc);
    const after = Math.max(0, subtotal - hdrDisc);
    const rounded = Math.round((after/0.05))*0.05;
    const adj = rounded - after;

    document.getElementById('t_subtotal').textContent = subtotal.toFixed(2);
    document.getElementById('t_hdrdisc').textContent  = hdrDisc.toFixed(2);
    document.getElementById('t_round').textContent    = adj.toFixed(2);
    document.getElementById('t_total').textContent    = rounded.toFixed(2);
  }
  ['f_discount_type','f_discount_value'].forEach(id=>{
    const el=document.getElementById(id); el && el.addEventListener('input', calcTotals);
  });

  function openRefund(id){ document.getElementById('refund_return_id').value=id; refundModal.show(); }

  // تحميل/فتح
  function openCreate(){ /* مُعرّفة أعلاه */ }
  function openEdit(id){ /* مُعرّفة أعلاه */ }
</script>
</body>
</html>
