<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* صلاحيات بسيطة (عدّل حسب نظامك) */
$canEdit = true;

/* CSRF */
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}
$csrf = $_SESSION['csrf_token'];

/* Helpers */
function clean_str(?string $s, int $max=255): ?string {
  $s = trim((string)$s);
  if ($s==='') return null;
  return mb_substr($s, 0, $max);
}
function slugify(?string $s): ?string {
  $s = trim((string)$s);
  if ($s==='') return null;
  $s = mb_strtolower($s,'UTF-8');
  $s = preg_replace('~[^\p{Arabic}a-z0-9\s\-]+~u', '', $s);
  $s = preg_replace('~[\s\-]+~u', '-', $s);
  $s = trim($s, '-');
  return $s ?: null;
}
function unique_slug(PDO $pdo, string $base, ?int $ignoreId=null): string {
  $slug = $base ?: ('prod-'.time());
  $i=2;
  while (1) {
    $sql = "SELECT 1 FROM products WHERE slug=:s".($ignoreId?" AND id<>:id":"")." LIMIT 1";
    $st = $pdo->prepare($sql);
    $p = [':s'=>$slug]; if ($ignoreId) $p[':id']=$ignoreId;
    $st->execute($p);
    if (!$st->fetch()) return $slug;
    $slug = $base.'-'.$i++;
  }
}
function buildTree(array $rows): array {
  $byId=[]; foreach ($rows as $r){ $r['children']=[]; $byId[$r['id']]=$r; }
  $root=[];
  foreach ($byId as $id=>&$n){
    if ($n['parent_id']) {
      if (isset($byId[$n['parent_id']])) $byId[$n['parent_id']]['children'][]=&$n;
      else $root[]=&$n;
    } else $root[]=&$n;
  }
  $sortFn=function(&$list) use (&$sortFn){
    usort($list,function($a,$b){
      if ((int)$a['sort_order']===(int)$b['sort_order'])
        return strnatcasecmp($a['name_ar'],$b['name_ar']);
      return (int)$a['sort_order'] <=> (int)$b['sort_order'];
    });
    foreach ($list as &$n) if (!empty($n['children'])) $sortFn($n['children']);
  };
  $sortFn($root);
  return $root;
}
function optionTreeHtml($tree, $level=0, $selected=null){
  $pad = str_repeat('— ', $level); $html='';
  foreach ($tree as $n){
    $sel = ($selected!==null && (int)$selected===(int)$n['id']) ? 'selected':'';
    $html .= '<option value="'.$n['id'].'" '.$sel.'>'.$pad.htmlspecialchars($n['name_ar']).'</option>';
    if (!empty($n['children'])) $html .= optionTreeHtml($n['children'], $level+1, $selected);
  }
  return $html;
}
function collectDescendants($tree, $wantedId, &$bag){
  foreach ($tree as $n){
    if ((int)$n['id'] === (int)$wantedId) {
      $stack = [$n];
      while ($stack){
        $x = array_pop($stack);
        $bag[] = (int)$x['id'];
        foreach ($x['children'] ?? [] as $ch) $stack[] = $ch;
      }
      return true;
    }
    if (!empty($n['children']) && collectDescendants($n['children'], $wantedId, $bag)) return true;
  }
  return false;
}

/* بيانات التصنيفات للأختيار */
$catRows = $pdo->query("SELECT * FROM categories WHERE deleted_at IS NULL ORDER BY sort_order, name_ar")->fetchAll(PDO::FETCH_ASSOC);
$catTree = buildTree($catRows);

/* ماركات */
$brands = $pdo->query("SELECT id, name FROM brands WHERE is_active=1 ORDER BY name")->fetchAll(PDO::FETCH_KEY_PAIR);

/* ==== Actions ==== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF). أعد تحميل الصفحة.';
  } elseif (!$canEdit) {
    $error = 'لا تملك صلاحية التعديل.';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action==='save') {
        $id          = ($_POST['id'] ?? '')!=='' ? (int)$_POST['id'] : null;
        $name_ar     = clean_str($_POST['name_ar'] ?? '', 160);
        $name_en     = clean_str($_POST['name_en'] ?? '', 160);
        $category_id = ($_POST['category_id'] ?? '')!=='' ? (int)$_POST['category_id'] : null;
        $brand_id    = ($_POST['brand_id'] ?? '')!=='' ? (int)$_POST['brand_id'] : null;
        $brand_new   = clean_str($_POST['brand_new'] ?? '', 120);
        $slug        = clean_str($_POST['slug'] ?? '', 180);
        $sku_prefix  = clean_str($_POST['sku_prefix'] ?? '', 40);
        $base_price  = (float)($_POST['base_price'] ?? 0);
        $compare     = ($_POST['compare_price'] ?? '')!=='' ? (float)$_POST['compare_price'] : null;
        $cost        = ($_POST['cost'] ?? '')!=='' ? (float)$_POST['cost'] : null;
        $vat_rate    = ($_POST['vat_rate'] ?? '')!=='' ? (float)$_POST['vat_rate'] : 0.00;
        $inc_vat     = isset($_POST['price_includes_vat']) ? 1 : 0;
        $tags        = clean_str($_POST['tags'] ?? '', 255);
        $desc        = clean_str($_POST['description'] ?? '', 10000);
        $active      = isset($_POST['is_active']) ? 1 : 0;
        $featured    = isset($_POST['is_featured']) ? 1 : 0;

        if (!$name_ar) throw new RuntimeException('اسم المنتج (عربي) مطلوب.');

        // إنشاء ماركة سريعة لو موجود نصًّا
        if ($brand_new) {
          $st = $pdo->prepare("SELECT id FROM brands WHERE name=?");
          $st->execute([$brand_new]);
          $bid = $st->fetchColumn();
          if (!$bid) {
            $pdo->prepare("INSERT INTO brands (name,is_active) VALUES (?,1)")->execute([$brand_new]);
            $bid = (int)$pdo->lastInsertId();
          }
          $brand_id = (int)$bid;
        }

        // slug فريد
        if (!$slug) $slug = slugify($name_en ?? '') ?: slugify($name_ar ?? '') ?: ('prod-'.time());
        $slug = unique_slug($pdo, $slug, $id);

        if ($id) {
          $sql = "UPDATE products SET category_id=:category_id, brand_id=:brand_id, name_ar=:name_ar, name_en=:name_en,
                  slug=:slug, sku_prefix=:sku_prefix, base_price=:base_price, compare_price=:compare_price,
                  cost=:cost, vat_rate=:vat_rate, price_includes_vat=:inc_vat, tags=:tags, description=:description,
                  is_active=:active, is_featured=:featured
                  WHERE id=:id AND deleted_at IS NULL";
          $st = $pdo->prepare($sql);
          $st->execute([
            ':category_id'=>$category_id, ':brand_id'=>$brand_id, ':name_ar'=>$name_ar, ':name_en'=>$name_en,
            ':slug'=>$slug, ':sku_prefix'=>$sku_prefix, ':base_price'=>$base_price, ':compare_price'=>$compare,
            ':cost'=>$cost, ':vat_rate'=>$vat_rate, ':inc_vat'=>$inc_vat, ':tags'=>$tags, ':description'=>$desc,
            ':active'=>$active, ':featured'=>$featured, ':id'=>$id
          ]);
          $success = '✔️ تم تحديث المنتج';
        } else {
          $st = $pdo->prepare("INSERT INTO products
            (category_id, brand_id, name_ar, name_en, slug, sku_prefix, base_price, compare_price, cost, vat_rate, price_includes_vat, tags, description, is_active, is_featured)
            VALUES (:category_id,:brand_id,:name_ar,:name_en,:slug,:sku_prefix,:base_price,:compare_price,:cost,:vat_rate,:inc_vat,:tags,:description,:active,:featured)");
          $st->execute([
            ':category_id'=>$category_id, ':brand_id'=>$brand_id, ':name_ar'=>$name_ar, ':name_en'=>$name_en, ':slug'=>$slug,
            ':sku_prefix'=>$sku_prefix, ':base_price'=>$base_price, ':compare_price'=>$compare, ':cost'=>$cost, ':vat_rate'=>$vat_rate,
            ':inc_vat'=>$inc_vat, ':tags'=>$tags, ':description'=>$desc, ':active'=>$active, ':featured'=>$featured
          ]);
          $success = '✔️ تم إضافة المنتج';
        }

        /* رفع صور (اختياري) */
        if (!empty($_FILES['images']) && is_array($_FILES['images']['name'])) {
          $uploadDir = __DIR__ . '/uploads/products/';
          if (!is_dir($uploadDir)) @mkdir($uploadDir,0777,true);
          $idNow = $id ?: (int)$pdo->lastInsertId();
          $allowed = ['jpg','jpeg','png','webp'];
          $c = $pdo->prepare("SELECT COALESCE(MAX(sort_order),0)+1 FROM product_images WHERE product_id=?");
          $c->execute([$idNow]); $sort = (int)$c->fetchColumn();
          for ($i=0; $i<count($_FILES['images']['name']); $i++){
            if (($_FILES['images']['error'][$i] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) continue;
            $ext = strtolower(pathinfo($_FILES['images']['name'][$i], PATHINFO_EXTENSION));
            if (!in_array($ext,$allowed,true)) continue;
            if ((int)$_FILES['images']['size'][$i] > 3*1024*1024) continue;
            $name = time().'_'.bin2hex(random_bytes(4)).'.'.$ext;
            if (move_uploaded_file($_FILES['images']['tmp_name'][$i], $uploadDir.$name)) {
              $pdo->prepare("INSERT INTO product_images (product_id,image,sort_order) VALUES (?,?,?)")
                  ->execute([$idNow,$name,$sort++]);
            }
          }
        }

      } elseif ($action==='toggle' && isset($_POST['id'])) {
        $id=(int)$_POST['id'];
        $pdo->prepare("UPDATE products SET is_active=1-is_active WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success='تم تغيير الحالة';

      } elseif ($action==='soft_delete' && isset($_POST['id'])) {
        $id=(int)$_POST['id'];
        $pdo->prepare("UPDATE products SET is_active=0, deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success='تم أرشفة المنتج';

      } elseif ($action==='restore' && isset($_POST['id'])) {
        $id=(int)$_POST['id'];
        $pdo->prepare("UPDATE products SET deleted_at=NULL, is_active=1 WHERE id=?")->execute([$id]);
        $success='تم الاستعادة';

      } elseif ($action==='reorder_images') {
        $pid = (int)($_POST['product_id'] ?? 0);
        $order = $_POST['order'] ?? [];
        if ($pid>0 && is_array($order)) {
          $i=0;
          $st = $pdo->prepare("UPDATE product_images SET sort_order=? WHERE id=? AND product_id=?");
          foreach ($order as $imgId) $st->execute([$i++,(int)$imgId,$pid]);
          $success='تم حفظ ترتيب الصور';
        }

      } elseif ($action==='delete_image') {
        $imgId = (int)($_POST['image_id'] ?? 0);
        $row = $pdo->prepare("SELECT product_id,image FROM product_images WHERE id=?");
        $row->execute([$imgId]); $img=$row->fetch(PDO::FETCH_ASSOC);
        if ($img) {
          $pdo->prepare("DELETE FROM product_images WHERE id=?")->execute([$imgId]);
          $path = __DIR__ . '/uploads/products/' . $img['image'];
          if (is_file($path)) @unlink($path);
          $success='تم حذف الصورة';
        }
      }
    } catch (Throwable $e) {
      $error = 'خطأ: '.htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
    }
  }
}

/* ==== فلاتر البحث ==== */
$q = trim($_GET['q'] ?? '');
$filterCat = ($_GET['category_id'] ?? '')!=='' ? (int)$_GET['category_id'] : null;
$includeChildren = isset($_GET['include_children']);
$filterBrand = ($_GET['brand_id'] ?? '')!=='' ? (int)$_GET['brand_id'] : null;
$showInactive = isset($_GET['show_inactive']);
$showArchived = isset($_GET['show_archived']);

/* بناء قائمة IDs للتصنيفات المُضمّنة */
$catIds = null;
if ($filterCat) {
  $catIds = [$filterCat];
  if ($includeChildren) collectDescendants($catTree, $filterCat, $catIds);
  $catIds = array_unique($catIds);
}

/* ==== جلب المنتجات ==== */
$where = [];
$params = [];
$where[] = $showArchived ? "1=1" : "p.deleted_at IS NULL";
if (!$showInactive) $where[] = "p.is_active=1";
if ($q!=='') { $where[]="(p.name_ar LIKE :q OR p.name_en LIKE :q OR p.slug LIKE :q OR p.tags LIKE :q)"; $params[':q']="%$q%"; }
if ($filterBrand) { $where[]="p.brand_id=:bid"; $params[':bid']=$filterBrand; }
if ($catIds) {
  $in = implode(',', array_fill(0, count($catIds), '?'));
  $where[] = "p.category_id IN ($in)";
  $params = array_merge($params, $catIds);
}
$whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';

$sql = "SELECT p.*,
               c.name_ar AS CategoryName,
               b.name AS BrandName,
               (SELECT COUNT(*) FROM product_variants v WHERE v.product_id=p.id AND v.deleted_at IS NULL) AS variants_count,
               (SELECT COALESCE(SUM(s.quantity),0)
                FROM product_variants v
                LEFT JOIN product_variant_stock s ON s.variant_id=v.id
                WHERE v.product_id=p.id AND v.deleted_at IS NULL) AS total_stock,
               (SELECT image FROM product_images WHERE product_id=p.id ORDER BY sort_order, id LIMIT 1) AS thumb
        FROM products p
        LEFT JOIN categories c ON c.id=p.category_id
        LEFT JOIN brands b ON b.id=p.brand_id
        $whereSql
        ORDER BY p.id DESC
        LIMIT 1000";
$st = $pdo->prepare($sql);
$st->execute(array_values($params));
$productsList = $st->fetchAll(PDO::FETCH_ASSOC);

/* صور لكل منتج (للمودال) */
$imgsMap = [];
if ($productsList) {
  $ids = array_column($productsList,'id');
  $in = implode(',', array_fill(0, count($ids), '?'));
  $st = $pdo->prepare("SELECT id, product_id, image, sort_order FROM product_images WHERE product_id IN ($in) ORDER BY sort_order, id");
  $st->execute($ids);
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $r) { $imgsMap[(int)$r['product_id']][] = $r; }
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>المنتجات</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#ffffff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{
      background: linear-gradient(90deg, var(--p1), var(--p2), var(--p3), var(--p4));
      padding:14px 18px; color:#1f2a37; display:flex; align-items:center; gap:12px; justify-content:space-between;
      box-shadow:0 6px 24px rgba(0,0,0,.15);
    }
    .user-info{display:flex; align-items:center; gap:10px; background:rgba(255,255,255,.6); padding:6px 10px; border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}

    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .img-thumb{width:48px;height:48px;object-fit:cover;border-radius:8px;border:1px solid var(--border)}
    .sortable-placeholder{border:2px dashed var(--primary); border-radius:12px; height:56px; margin:8px 0}
    .muted{color:var(--muted)}
    .table>:not(caption)>*>*{background:transparent}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-shirt me-2"></i> المنتجات</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">
  <!-- شريط الأدوات: بحث وفلاتر + زر إضافة -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث بالاسم/السلَج/الوسوم" value="<?= htmlspecialchars($q) ?>">
        <select class="form-select" name="category_id" style="min-width:220px">
          <option value="">كل التصنيفات</option>
          <?= optionTreeHtml($catTree, 0, $filterCat) ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="include_children" name="include_children" value="1" <?= $includeChildren?'checked':''; ?>>
          <label class="form-check-label" for="include_children">تضمين الأبناء</label>
        </div>
        <select class="form-select" name="brand_id" style="min-width:200px">
          <option value="">كل الماركات</option>
          <?php foreach ($brands as $bid=>$bname): ?>
            <option value="<?= (int)$bid ?>" <?= $filterBrand===$bid?'selected':''; ?>><?= htmlspecialchars($bname) ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_inactive" name="show_inactive" value="1" <?= $showInactive?'checked':''; ?>>
          <label class="form-check-label" for="show_inactive">إظهار غير المفعل</label>
        </div>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $filterCat || $filterBrand || $showInactive || $showArchived): ?>
          <a href="products.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>
      <?php if ($canEdit): ?>
      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#productModal" onclick="openCreate()">
        <i class="fa-solid fa-plus"></i> منتج جديد
      </button>
      <?php endif; ?>
    </div>
  </div>

  <!-- جدول المنتجات -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>صورة</th>
              <th class="text-start">الاسم</th>
              <th>التصنيف</th>
              <th>الماركة</th>
              <th>السعر</th>
              <th>المتغيّرات</th>
              <th>إجمالي المخزون</th>
              <th>الحالة</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!$productsList): ?>
              <tr><td colspan="10" class="text-muted">لا توجد بيانات.</td></tr>
            <?php else: foreach ($productsList as $p): 
              $pid = (int)$p['id'];
              $imgs = $imgsMap[$pid] ?? [];
              $thumb = $p['thumb'] ? 'uploads/products/'.htmlspecialchars($p['thumb']) : null;
            ?>
              <tr data-pid="<?= $pid ?>" 
                  data-product='<?= htmlspecialchars(json_encode($p), ENT_QUOTES, "UTF-8") ?>'
                  data-images='<?= htmlspecialchars(json_encode($imgs), ENT_QUOTES, "UTF-8") ?>'>
                <td><?= $pid ?></td>
                <td>
                  <?php if ($thumb): ?>
                    <img class="img-thumb" src="<?= $thumb ?>" alt="">
                  <?php else: ?>
                    <span class="muted">—</span>
                  <?php endif; ?>
                </td>
                <td class="text-start">
                  <div class="fw-bold"><?= htmlspecialchars($p['name_ar'] ?? '—', ENT_QUOTES, 'UTF-8') ?></div>
                  <div class="muted small"><?= htmlspecialchars($p['name_en'] ?? '—', ENT_QUOTES, 'UTF-8') ?> — <span class="pill"><?= htmlspecialchars($p['slug'] ?? '—', ENT_QUOTES, 'UTF-8') ?></span></div>
                  <?php if ((int)$p['is_featured']===1): ?><span class="badge bg-info text-dark mt-1">مميّز</span><?php endif; ?>
                </td>
                <td><?= htmlspecialchars($p['CategoryName'] ?? '—', ENT_QUOTES, 'UTF-8') ?></td>
                <td><?= htmlspecialchars($p['BrandName'] ?? '—', ENT_QUOTES, 'UTF-8') ?></td>
                <td>
                  <?= number_format((float)$p['base_price'],2) ?>
                  <?php if ($p['compare_price'] !== null): ?>
                    <div class="muted"><s><?= number_format((float)$p['compare_price'],2) ?></s></div>
                  <?php endif; ?>
                </td>
                <td><span class="pill"><?= (int)($p['variants_count'] ?? 0) ?></span></td>
                <td><?= (int)($p['total_stock'] ?? 0) ?></td>
                <td><?= ((int)$p['is_active']===1) ? '<span class="badge bg-success">مفعل</span>' : '<span class="badge bg-warning text-dark">غير مفعل</span>' ?></td>
                <td>
                  <div class="btn-group">
                    <a class="btn btn-sm btn-outline-primary" href="variants.php?product_id=<?= $pid ?>"><i class="fa-solid fa-palette"></i></a>
                    <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(this)"><i class="fa-solid fa-pen-to-square"></i></button>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="toggle">
                      <input type="hidden" name="id" value="<?= $pid ?>">
                      <button class="btn btn-sm btn-outline-warning"><i class="fa-solid fa-toggle-on"></i></button>
                    </form>
                    <?php if ($p['deleted_at'] === null): ?>
                      <form method="post" class="d-inline" onsubmit="return confirm('أرشفة المنتج؟')">
                        <input type="hidden" name="csrf" value="<?= $csrf ?>">
                        <input type="hidden" name="action" value="soft_delete">
                        <input type="hidden" name="id" value="<?= $pid ?>">
                        <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                      </form>
                    <?php else: ?>
                      <form method="post" class="d-inline">
                        <input type="hidden" name="csrf" value="<?= $csrf ?>">
                        <input type="hidden" name="action" value="restore">
                        <input type="hidden" name="id" value="<?= $pid ?>">
                        <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                      </form>
                    <?php endif; ?>
                  </div>
                </td>
              </tr>
            <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<!-- Modal: Create/Edit Product -->
<div class="modal fade" id="productModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-pen-to-square me-2"></i> <span id="modalTitle">منتج جديد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <form method="post" enctype="multipart/form-data" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save">
        <input type="hidden" name="id" id="f_id">
        <div class="row g-3">

          <div class="col-md-6">
            <label class="form-label">الاسم (عربي) *</label>
            <input type="text" class="form-control" name="name_ar" id="f_name_ar" required>
          </div>
          <div class="col-md-6">
            <label class="form-label">الاسم (إنجليزي)</label>
            <input type="text" class="form-control" name="name_en" id="f_name_en">
          </div>

          <div class="col-md-4">
            <label class="form-label">التصنيف</label>
            <select class="form-select" name="category_id" id="f_category">
              <option value="">— اختر —</option>
              <?= optionTreeHtml($catTree) ?>
            </select>
          </div>

          <div class="col-md-4">
            <label class="form-label">الماركة</label>
            <select class="form-select" name="brand_id" id="f_brand">
              <option value="">— اختر —</option>
              <?php foreach ($brands as $bid=>$bname): ?>
                <option value="<?= (int)$bid ?>"><?= htmlspecialchars($bname) ?></option>
              <?php endforeach; ?>
            </select>
            <div class="form-text">أو أضف ماركة جديدة:</div>
            <input type="text" class="form-control mt-1" name="brand_new" id="f_brand_new" placeholder="أدخل اسم الماركة">
          </div>

          <div class="col-md-4">
            <label class="form-label">Slug (اختياري)</label>
            <input type="text" class="form-control" name="slug" id="f_slug" placeholder="لو تركته فارغًا سيُولّد تلقائيًا">
          </div>

          <div class="col-md-3">
            <label class="form-label">SKU Prefix (للمتغيّرات)</label>
            <input type="text" class="form-control" name="sku_prefix" id="f_sku_prefix">
          </div>
          <div class="col-md-3">
            <label class="form-label">السعر الأساسي</label>
            <input type="number" step="0.01" class="form-control" name="base_price" id="f_base_price" value="0">
          </div>
          <div class="col-md-3">
            <label class="form-label">سعر مقارنة</label>
            <input type="number" step="0.01" class="form-control" name="compare_price" id="f_compare">
          </div>
          <div class="col-md-3">
            <label class="form-label">التكلفة</label>
            <input type="number" step="0.01" class="form-control" name="cost" id="f_cost">
          </div>

          <div class="col-md-2">
            <label class="form-label">% الضريبة</label>
            <input type="number" step="0.01" class="form-control" name="vat_rate" id="f_vat" value="0">
          </div>
          <div class="col-md-2 d-flex align-items-end">
            <div class="form-check">
              <input class="form-check-input" type="checkbox" name="price_includes_vat" id="f_incvat">
              <label class="form-check-label" for="f_incvat">السعر شامل الضريبة</label>
            </div>
          </div>
          <div class="col-md-8">
            <label class="form-label">الوسوم (Tags)</label>
            <input type="text" class="form-control" name="tags" id="f_tags" placeholder="مثال: صيفي، قطن، أوفر سايز">
          </div>

          <div class="col-12">
            <label class="form-label">الوصف</label>
            <textarea class="form-control" name="description" id="f_desc" rows="3"></textarea>
          </div>

          <div class="col-md-6">
            <label class="form-label">صور المنتج (متعددة، حتى 3MB للصورة)</label>
            <input type="file" class="form-control" name="images[]" multiple accept=".jpg,.jpeg,.png,.webp,image/*">
          </div>

          <div class="col-md-3 d-flex align-items-end">
            <div class="form-check">
              <input class="form-check-input" type="checkbox" name="is_active" id="f_active" checked>
              <label class="form-check-label" for="f_active">مفعل</label>
            </div>
          </div>
          <div class="col-md-3 d-flex align-items-end">
            <div class="form-check">
              <input class="form-check-input" type="checkbox" name="is_featured" id="f_featured">
              <label class="form-check-label" for="f_featured">مميّز</label>
            </div>
          </div>

          <div class="col-12">
            <div class="d-flex flex-wrap gap-2" id="img_list"><!-- صور المنتج تظهر هنا --></div>
          </div>

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
            <a class="btn btn-outline-secondary ms-2" id="goVariantsBtn" href="#" target="_self">
              <i class="fa-solid fa-palette"></i> إدارة المتغيّرات
            </a>
          </div>
        </div>
      </form>

      <!-- نموذج داخلي لإجراءات الصور -->
      <form id="img_actions" method="post" class="d-none">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="">
        <input type="hidden" name="product_id" value="">
        <input type="hidden" name="order[]" value="">
        <input type="hidden" name="image_id" value="">
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js"></script>
<script>
  // Dark mode
  const body = document.body;
  const darkSwitch = document.getElementById("darkSwitch");
  if (localStorage.getItem("darkMode")==="true"){ body.classList.add("dark"); darkSwitch.textContent="☀️"; }
  darkSwitch?.addEventListener("click", ()=> {
    body.classList.toggle("dark");
    const isDark = body.classList.contains("dark");
    localStorage.setItem("darkMode", isDark);
    darkSwitch.textContent = isDark ? "☀️" : "🌙";
  });

  const productModal = new bootstrap.Modal(document.getElementById('productModal'));
  const modalTitle = document.getElementById('modalTitle');

  function openCreate(){
    modalTitle.textContent='منتج جديد';
    document.getElementById('f_id').value='';
    ['f_name_ar','f_name_en','f_slug','f_sku_prefix','f_compare','f_cost','f_tags','f_desc'].forEach(id=>document.getElementById(id).value='');
    document.getElementById('f_base_price').value='0';
    document.getElementById('f_vat').value='0';
    document.getElementById('f_category').value='';
    document.getElementById('f_brand').value='';
    document.getElementById('f_brand_new').value='';
    document.getElementById('f_active').checked=true;
    document.getElementById('f_featured').checked=false;
    document.getElementById('img_list').innerHTML='';
    document.getElementById('goVariantsBtn').href='#';
    productModal.show();
  }

  function openEdit(btn){
    modalTitle.textContent='تعديل منتج';
    const tr = btn.closest('tr');
    const p  = JSON.parse(tr.getAttribute('data-product'));
    const images = JSON.parse(tr.getAttribute('data-images') || '[]');

    document.getElementById('f_id').value = p.id ?? '';
    document.getElementById('f_name_ar').value = p.name_ar ?? '';
    document.getElementById('f_name_en').value = p.name_en ?? '';
    document.getElementById('f_slug').value = p.slug ?? '';
    document.getElementById('f_sku_prefix').value = p.sku_prefix ?? '';
    document.getElementById('f_base_price').value = p.base_price ?? 0;
    document.getElementById('f_compare').value = p.compare_price ?? '';
    document.getElementById('f_cost').value = p.cost ?? '';
    document.getElementById('f_vat').value = p.vat_rate ?? 0;
    document.getElementById('f_incvat').checked = (parseInt(p.price_includes_vat ?? 0)===1);
    document.getElementById('f_tags').value = p.tags ?? '';
    document.getElementById('f_desc').value = p.description ?? '';
    document.getElementById('f_category').value = p.category_id ?? '';
    document.getElementById('f_brand').value = p.brand_id ?? '';
    document.getElementById('f_brand_new').value = '';
    document.getElementById('f_active').checked = (parseInt(p.is_active ?? 1)===1);
    document.getElementById('f_featured').checked = (parseInt(p.is_featured ?? 0)===1);

    // زر إدارة المتغيّرات
    document.getElementById('goVariantsBtn').href = 'variants.php?product_id='+(p.id ?? '');

    // صور
    const imgList = document.getElementById('img_list');
    imgList.innerHTML='';
    images.forEach(im=>{
      const el = document.createElement('div');
      el.className='d-flex flex-column align-items-center';
      el.setAttribute('data-imgid', im.id);
      el.innerHTML = `
        <img class="img-thumb" src="uploads/products/${im.image}" alt="">
        <button type="button" class="btn btn-sm btn-outline-danger mt-1" onclick="deleteImage(${im.id})"><i class="fa-solid fa-trash"></i></button>
      `;
      imgList.appendChild(el);
    });

    // Sortable images
    new Sortable(imgList,{
      animation:150,
      onEnd: function(){
        const ids = Array.from(imgList.children).map(el=>el.getAttribute('data-imgid'));
        const fd = new FormData();
        fd.append('csrf','<?= $csrf ?>');
        fd.append('action','reorder_images');
        fd.append('product_id', p.id);
        ids.forEach(id=>fd.append('order[]', id));
        fetch(location.href,{method:'POST', body:fd});
      }
    });

    productModal.show();
  }

  function deleteImage(imageId){
    if (!confirm('حذف الصورة؟')) return;
    const fd = new FormData();
    fd.append('csrf','<?= $csrf ?>');
    fd.append('action','delete_image');
    fd.append('image_id', imageId);
    fetch(location.href,{method:'POST', body:fd})
      .then(()=> location.reload());
  }
</script>
</body>
</html>
