<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* صلاحيات */
$canEdit = true;

/* CSRF */
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf_token'];

/* ===== Helpers عامة ===== */
if (!function_exists('hasColumn')) {
  function hasColumn(PDO $pdo, string $table, string $col): bool {
    $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name=? AND column_name=?");
    $st->execute([$table,$col]);
    return (int)$st->fetchColumn() > 0;
  }
}
function supplierNameCol(PDO $pdo): string {
  foreach (['name','SupplierName','supplier_name','company_name','CompanyName'] as $c) {
    if (hasColumn($pdo,'suppliers',$c)) return $c;
  }
  return 'name';
}
$SUPPLIER_NAME_COL = supplierNameCol($pdo);

function clean_str(?string $s, int $max=255): ?string { $s=trim((string)$s); if($s==='') return null; return mb_substr($s,0,$max); }
function money($n){ return number_format((float)$n,2); }

/* فروع */
try {
  $branches = $pdo->query("SELECT BranchID, BranchName FROM Branches ORDER BY BranchName")->fetchAll(PDO::FETCH_KEY_PAIR);
} catch(Throwable $e){ $branches = []; }

/* موردون (يدعم غياب is_active) */
$hasSupActive = hasColumn($pdo,'suppliers','is_active');
$sqlSup = "SELECT id, `{$SUPPLIER_NAME_COL}` AS name FROM suppliers"
        . ($hasSupActive ? " WHERE is_active=1" : "")
        . " ORDER BY `{$SUPPLIER_NAME_COL}`";
$suppliers = $pdo->query($sqlSup)->fetchAll(PDO::FETCH_KEY_PAIR);

/* متغيّرات للبحث السريع */
$variants = $pdo->query("
  SELECT v.id, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-'),' | ',v.sku) AS label
  FROM product_variants v
  LEFT JOIN products p ON p.id=v.product_id
  WHERE v.deleted_at IS NULL
  ORDER BY p.name_ar, v.id
  LIMIT 2000
")->fetchAll(PDO::FETCH_ASSOC);

/* ===== تحميل عملية واحدة (JSON endpoint) ===== */
function load_return(PDO $pdo, int $id){
  $h = $pdo->prepare("SELECT * FROM purchase_returns WHERE id=?");
  $h->execute([$id]); $header = $h->fetch(PDO::FETCH_ASSOC);
  if (!$header) return [null, []];
  $i = $pdo->prepare("
    SELECT ri.*, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM purchase_return_items ri
    JOIN product_variants v ON v.id=ri.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE ri.return_id=?
    ORDER BY ri.id
  ");
  $i->execute([$id]);
  return [$header, $i->fetchAll(PDO::FETCH_ASSOC)];
}

/* Endpoint JSON مبكر قبل أي HTML */
if (isset($_GET['getReturn'])) {
  header('Content-Type: application/json; charset=utf-8');
  $id = (int)$_GET['getReturn'];
  [$h,$it] = load_return($pdo,$id);
  echo json_encode(['header'=>$h,'items'=>$it], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== حساب الإجماليات (سيرفر) ===== */
function calc_totals(array $items, string $discount_type, float $discount_value, float $tax_rate, float $other_fees){
  $subtotal = 0.0;
  foreach ($items as $it){
    $line = max(0,(int)($it['qty'] ?? 0)) * (float)$it['unit_cost'] - (float)$it['line_discount'];
    $line += $line * ((float)($it['line_tax_rate'] ?? 0)/100);
    $subtotal += $line;
  }
  $hdrDisc = ($discount_type==='percent') ? ($subtotal * ($discount_value/100)) : $discount_value;
  $after = max(0, $subtotal - $hdrDisc);
  $hdrTax = $after * ($tax_rate/100);
  $total = $after + $hdrTax + $other_fees;
  return [$subtotal, $hdrDisc, $hdrTax, $total];
}

/* ===== تنفيذ الحركة المخزنية عند الإرجاع =====
   نستخدم processed_qty للتنفيذ المرحلي. نسجل الفارق في المخزون والحركة. */
function return_and_move(PDO $pdo, int $return_id, int $branch_id){
  $stmt = $pdo->prepare("SELECT id, variant_id, qty, processed_qty, unit_cost FROM purchase_return_items WHERE return_id=? ORDER BY id");
  $stmt->execute([$return_id]);
  $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
  foreach ($items as $it){
    $delta = (int)$it['processed_qty'] - (int)($GLOBALS['_prev_processed'][$it['id']] ?? 0);
    if ($delta > 0 && $branch_id) {
      // خصم من مخزون الفرع
      $up = $pdo->prepare("
        INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
        VALUES (?,?,?,0)
        ON DUPLICATE KEY UPDATE quantity = GREATEST(0, quantity - VALUES(quantity))
      ");
      $up->execute([(int)$it['variant_id'], $branch_id, $delta]);

      // حركة مخزون سالبة
      $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                     VALUES (?,?,?,?, 'purchase_return', ?)")->execute([
        (int)$it['variant_id'], $branch_id, -$delta, (float)$it['unit_cost'], $return_id
      ]);
    }
  }
}

/* للاحتفاظ بالقيم السابقة قبل التحديث */
$_prev_processed = [];

/* ===== POST Actions ===== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF).';
  } elseif (!$canEdit) {
    $error = 'لا تملك صلاحية.';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action === 'save_return') {
        $id           = ($_POST['id'] ?? '')!=='' ? (int)$_POST['id'] : null;
        $supplier_id  = (int)($_POST['supplier_id'] ?? 0);
        $branch_id    = ($_POST['branch_id'] ?? '')!=='' ? (int)$_POST['branch_id'] : null;
        $purchase_id  = ($_POST['purchase_id'] ?? '')!=='' ? (int)$_POST['purchase_id'] : null;
        $status       = $_POST['status'] ?? 'draft';
        $return_date  = $_POST['return_date'] ?: date('Y-m-d');
        $currency     = clean_str($_POST['currency_code'] ?? 'EGP', 10);
        $discount_type= $_POST['discount_type'] ?? 'fixed';
        $discount_val = (float)($_POST['discount_value'] ?? 0);
        $tax_rate     = (float)($_POST['tax_rate'] ?? 0);
        $other_fees   = (float)($_POST['other_fees'] ?? 0);
        $notes        = clean_str($_POST['notes'] ?? '', 10000);

        // بنود
        $var_ids   = array_map('intval', $_POST['item_variant_id'] ?? []);
        $desc_arr  = $_POST['item_desc'] ?? [];
        $qty_arr   = array_map('intval', $_POST['item_qty'] ?? []);
        $proc_arr  = array_map('intval', $_POST['item_processed_qty'] ?? []);
        $cost_arr  = array_map('floatval', $_POST['item_unit_cost'] ?? []);
        $disc_arr  = array_map('floatval', $_POST['item_line_discount'] ?? []);
        $tax_arr   = array_map('floatval', $_POST['item_line_tax'] ?? []);

        $items = [];
        for($i=0;$i<count($var_ids);$i++){
          if ($var_ids[$i] <= 0) continue;
          $qty  = max(0, (int)($qty_arr[$i] ?? 0));
          $proc = max(0, (int)($proc_arr[$i] ?? 0));
          if ($proc > $qty) $proc = $qty; // حمايه
          $unit = max(0.0,(float)($cost_arr[$i] ?? 0));
          $ld   = max(0.0,(float)($disc_arr[$i] ?? 0));
          $lt   = max(0.0,(float)($tax_arr[$i] ?? 0));
          $desc = clean_str($desc_arr[$i] ?? '', 255);
          if ($qty===0 && $proc===0) continue;
          $items[] = [
            'variant_id'=>$var_ids[$i], 'description'=>$desc,
            'qty'=>$qty, 'processed_qty'=>$proc,
            'unit_cost'=>$unit, 'line_discount'=>$ld, 'line_tax_rate'=>$lt
          ];
        }
        if ($supplier_id<=0) throw new RuntimeException('اختر موردًا.');
        if (!$items) throw new RuntimeException('أضف بندًا واحدًا على الأقل.');

        // إجماليات
        [$subtotal,$hdrDisc,$hdrTax,$total] = calc_totals($items, $discount_type, $discount_val, $tax_rate, $other_fees);

        // مرفق
        $attachName=null;
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error']===UPLOAD_ERR_OK){
          $dir = __DIR__ . '/uploads/purchase_returns/';
          if (!is_dir($dir)) @mkdir($dir,0777,true);
          $ext = strtolower(pathinfo($_FILES['attachment']['name'], PATHINFO_EXTENSION));
          $name = time().'_'.bin2hex(random_bytes(4)).'.'.$ext;
          if (move_uploaded_file($_FILES['attachment']['tmp_name'], $dir.$name)) $attachName=$name;
        }

        if ($id){
          // خزّن processed السابق
          $pr = $pdo->prepare("SELECT id, processed_qty FROM purchase_return_items WHERE return_id=?");
          $pr->execute([$id]); foreach($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_processed[(int)$r['id']] = (int)$r['processed_qty']; }

          $sql = "UPDATE purchase_returns SET supplier_id=:supplier_id, branch_id=:branch_id, purchase_id=:purchase_id,
                  status=:status, return_date=:return_date, currency_code=:currency_code, subtotal=:subtotal,
                  discount_type=:discount_type, discount_value=:discount_value, tax_rate=:tax_rate, other_fees=:other_fees,
                  total=:total, notes=:notes".($attachName? ", attachment=:attachment":"")." WHERE id=:id AND deleted_at IS NULL";
          $st = $pdo->prepare($sql);
          $p = [
            ':supplier_id'=>$supplier_id, ':branch_id'=>$branch_id, ':purchase_id'=>$purchase_id,
            ':status'=>$status, ':return_date'=>$return_date, ':currency_code'=>$currency, ':subtotal'=>$subtotal,
            ':discount_type'=>$discount_type, ':discount_value'=>$discount_val, ':tax_rate'=>$tax_rate,
            ':other_fees'=>$other_fees, ':total'=>$total, ':notes'=>$notes, ':id'=>$id
          ];
          if ($attachName) $p[':attachment']=$attachName;
          $st->execute($p);

          // إعادة إدخال البنود
          $pdo->prepare("DELETE FROM purchase_return_items WHERE return_id=?")->execute([$id]);
          $ins = $pdo->prepare("INSERT INTO purchase_return_items
              (return_id, variant_id, description, qty, processed_qty, unit_cost, line_discount, line_tax_rate, line_total)
              VALUES (?,?,?,?,?,?,?,?,?)");
          foreach ($items as $it){
            $line = ($it['qty']*$it['unit_cost']) - $it['line_discount'];
            $line += $line * ($it['line_tax_rate']/100);
            $ins->execute([$id,$it['variant_id'],$it['description'],$it['qty'],$it['processed_qty'],$it['unit_cost'],$it['line_discount'],$it['line_tax_rate'],$line]);
          }

          // تنفيذ مخزون لو زاد processed أو الحالة partial/returned
          if (in_array($status,['partial','returned'],true) && $branch_id){
            return_and_move($pdo, $id, $branch_id);
          }

          $success = '✔️ تم تحديث المرتجع';

        } else {
          // إنشاء رأس
          $st = $pdo->prepare("INSERT INTO purchase_returns
            (supplier_id, branch_id, purchase_id, status, return_date, currency_code, subtotal, discount_type, discount_value, tax_rate, other_fees, total, notes, attachment)
            VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
          $st->execute([$supplier_id,$branch_id,$purchase_id,$status,$return_date,$currency,$subtotal,$discount_type,$discount_val,$tax_rate,$other_fees,$total,$notes,$attachName]);
          $rid = (int)$pdo->lastInsertId();

          // إدخال البنود
          $ins = $pdo->prepare("INSERT INTO purchase_return_items
            (return_id, variant_id, description, qty, processed_qty, unit_cost, line_discount, line_tax_rate, line_total)
            VALUES (?,?,?,?,?,?,?,?,?)");
          foreach ($items as $it){
            $line = ($it['qty']*$it['unit_cost']) - $it['line_discount'];
            $line += $line * ($it['line_tax_rate']/100);
            $ins->execute([$rid,$it['variant_id'],$it['description'],$it['qty'],$it['processed_qty'],$it['unit_cost'],$it['line_discount'],$it['line_tax_rate'],$line]);
          }

          // تنفيذ فوري لو الحالة partial/returned
          if (in_array($status,['partial','returned'],true) && $branch_id){
            $_prev_processed = []; // جديد = 0
            return_and_move($pdo, $rid, $branch_id);
          }

          $success = '✔️ تم إنشاء المرتجع';
        }

      } elseif ($action==='execute_all') {
        $id = (int)($_POST['id'] ?? 0);
        if ($id<=0) throw new RuntimeException('معرّف غير صالح.');
        // خزّن السابق
        $pr = $pdo->prepare("SELECT id, processed_qty FROM purchase_return_items WHERE return_id=?");
        $pr->execute([$id]); foreach($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_processed[(int)$r['id']] = (int)$r['processed_qty']; }
        // اجعل المنفّذ = المطلوب لكل البنود
        $pdo->prepare("UPDATE purchase_return_items SET processed_qty = qty WHERE return_id=?")->execute([$id]);
        // الحالة returned
        $pdo->prepare("UPDATE purchase_returns SET status='returned' WHERE id=?")->execute([$id]);
        // فرع العملية
        $br = $pdo->prepare("SELECT branch_id FROM purchase_returns WHERE id=?");
        $br->execute([$id]); $branch_id = (int)$br->fetchColumn();
        if ($branch_id) return_and_move($pdo, $id, $branch_id);
        $success = '✔️ تم تنفيذ المرتجع بالكامل';

      } elseif ($action==='add_refund') {
        $rid = (int)($_POST['id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $method = clean_str($_POST['method'] ?? '', 40);
        $ref_no = clean_str($_POST['ref_no'] ?? '', 80);
        $notes  = clean_str($_POST['refund_notes'] ?? '', 255);
        if ($rid<=0 || $amount<=0) throw new RuntimeException('مبلغ غير صالح.');
        $st = $pdo->prepare("SELECT supplier_id FROM purchase_returns WHERE id=?");
        $st->execute([$rid]); $sup = (int)$st->fetchColumn();
        if ($sup<=0) throw new RuntimeException('مورد غير معروف.');
        $pdo->prepare("INSERT INTO supplier_refunds (supplier_id, return_id, amount, method, ref_no, notes)
                       VALUES (?,?,?,?,?,?)")->execute([$sup,$rid,$amount,$method,$ref_no,$notes]);
        $success = '✔️ تم تسجيل الاسترداد من المورد';

      } elseif ($action==='soft_delete') {
        $id = (int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE purchase_returns SET deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم أرشفة المرتجع';

      } elseif ($action==='restore') {
        $id = (int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE purchase_returns SET deleted_at=NULL WHERE id=?")->execute([$id]);
        $success = 'تم استعادة المرتجع';
      }

    } catch(Throwable $e){
      $error = 'خطأ: '.htmlspecialchars($e->getMessage(),ENT_QUOTES,'UTF-8');
    }
  }
}

/* ===== فلاتر العرض ===== */
$q            = trim($_GET['q'] ?? '');
$fSupplier    = ($_GET['supplier_id'] ?? '')!=='' ? (int)$_GET['supplier_id'] : null;
$fBranch      = ($_GET['branch_id'] ?? '')!=='' ? (int)$_GET['branch_id'] : null;
$fStatus      = $_GET['status'] ?? '';
$showArchived = isset($_GET['show_archived']);

$where=[]; $params=[];
$where[] = $showArchived ? "1=1" : "r.deleted_at IS NULL";
if ($q!==''){ $where[]="(r.id LIKE :q OR s.`{$SUPPLIER_NAME_COL}` LIKE :q)"; $params[':q']="%$q%"; }
if ($fSupplier){ $where[]="r.supplier_id=:sid"; $params[':sid']=$fSupplier; }
if ($fBranch){ $where[]="r.branch_id=:bid"; $params[':bid']=$fBranch; }
if ($fStatus!==''){ $where[]="r.status=:st"; $params[':st']=$fStatus; }
$whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';

$sql = "SELECT r.*, s.`{$SUPPLIER_NAME_COL}` AS SupplierName, b.BranchName
        FROM purchase_returns r
        LEFT JOIN suppliers s ON s.id=r.supplier_id
        LEFT JOIN Branches b ON b.BranchID=r.branch_id
        $whereSql
        ORDER BY r.id DESC
        LIMIT 500";
$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

/* تجميع استردادات لكل مرتجع لعرض الرصيد المتبقي */
$refundMap = [];
if ($rows){
  $ids = array_column($rows,'id');
  $in = implode(',', array_fill(0,count($ids),'?'));
  $ps = $pdo->prepare("SELECT return_id, SUM(amount) as sum_amount FROM supplier_refunds WHERE return_id IN ($in) GROUP BY return_id");
  $ps->execute($ids);
  foreach ($ps->fetchAll(PDO::FETCH_ASSOC) as $x){
    $refundMap[(int)$x['return_id']] = (float)$x['sum_amount'];
  }
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>مرتجعات الشراء</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#fff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{ background:linear-gradient(90deg,var(--p1),var(--p2),var(--p3),var(--p4)); padding:14px 18px; display:flex; align-items:center; justify-content:space-between; box-shadow:0 6px 24px rgba(0,0,0,.15); }
    .user-info{display:flex;align-items:center;gap:10px;background:rgba(255,255,255,.6);padding:6px 10px;border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}
    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .muted{color:var(--muted)}
    .num{min-width:90px;text-align:center}
    .table>:not(caption)>*>*{background:transparent}
    .tot-box{background:rgba(0,0,0,.03); border-radius:12px; padding:10px}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-boxes-packing me-2"></i> مرتجعات الشراء</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">
  <!-- فلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث برقم المرتجع/المورد" value="<?= htmlspecialchars($q) ?>">
        <select class="form-select" name="supplier_id" style="min-width:200px">
          <option value="">كل الموردين</option>
          <?php foreach ($suppliers as $sid=>$sname): ?>
            <option value="<?= (int)$sid ?>" <?= $fSupplier===$sid?'selected':''; ?>><?= htmlspecialchars($sname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="branch_id" style="min-width:200px">
          <option value="">كل الفروع</option>
          <?php foreach ($branches as $bid=>$bname): ?>
            <option value="<?= (int)$bid ?>" <?= $fBranch===$bid?'selected':''; ?>><?= htmlspecialchars($bname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="status" style="min-width:200px">
          <option value="">كل الحالات</option>
          <?php foreach (['draft'=>'مسودة','requested'=>'معتمدة','partial'=>'تنفيذ جزئي','returned'=>'تم التنفيذ','cancelled'=>'ملغي'] as $k=>$v): ?>
            <option value="<?= $k ?>" <?= $fStatus===$k?'selected':''; ?>><?= $v ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $fSupplier || $fBranch || $fStatus || $showArchived): ?>
          <a href="purchase_returns.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>

      <?php if ($canEdit): ?>
      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#returnModal" onclick="openCreate()">
        <i class="fa-solid fa-plus"></i> مرتجع جديد
      </button>
      <?php endif; ?>
    </div>
  </div>

  <!-- جدول -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>التاريخ</th>
              <th>المورد</th>
              <th>الفرع</th>
              <th>الحالة</th>
              <th>الإجمالي</th>
              <th>مُسترد</th>
              <th>متبقي</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="9" class="text-muted">لا توجد بيانات.</td></tr>
          <?php else: foreach ($rows as $r): 
              $refunded = (float)($refundMap[(int)$r['id']] ?? 0);
              $due = (float)$r['total'] - $refunded;
          ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>
              <td><?= htmlspecialchars($r['return_date']) ?></td>
              <td><?= htmlspecialchars($r['SupplierName'] ?? '-') ?></td>
              <td><?= htmlspecialchars($r['BranchName'] ?? '-') ?></td>
              <td>
                <?php
                  $badges = [
                    'draft'=>'<span class="badge bg-secondary">مسودة</span>',
                    'requested'=>'<span class="badge bg-info text-dark">معتمدة</span>',
                    'partial'=>'<span class="badge bg-warning text-dark">تنفيذ جزئي</span>',
                    'returned'=>'<span class="badge bg-success">تم التنفيذ</span>',
                    'cancelled'=>'<span class="badge bg-dark">ملغي</span>'
                  ];
                  echo $badges[$r['status']] ?? $r['status'];
                ?>
              </td>
              <td><?= money($r['total']) ?></td>
              <td><?= money($refunded) ?></td>
              <td><?= money($due) ?></td>
              <td>
                <div class="btn-group">
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <?php if ($r['status']!=='returned' && $r['status']!=='cancelled'): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('تنفيذ المرتجع بالكامل؟ سيُخصم المخزون.')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="execute_all">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-dolly"></i></button>
                    </form>
                  <?php endif; ?>
                  <button class="btn btn-sm btn-outline-primary" onclick="openRefund(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-receipt"></i></button>
                  <?php if ($r['deleted_at'] === null): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('أرشفة المرتجع؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="soft_delete">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                    </form>
                  <?php else: ?>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="restore">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                    </form>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<!-- Modal: Create/Edit Return -->
<div class="modal fade" id="returnModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-box-open me-2"></i> <span id="modalTitle">مرتجع جديد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>

      <form method="post" enctype="multipart/form-data" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save_return">
        <input type="hidden" name="id" id="f_id">

        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">المورد *</label>
            <select class="form-select" name="supplier_id" id="f_supplier" required>
              <option value="">— اختر —</option>
              <?php foreach ($suppliers as $sid=>$sname): ?>
                <option value="<?= (int)$sid ?>"><?= htmlspecialchars($sname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">الفرع</label>
            <select class="form-select" name="branch_id" id="f_branch">
              <option value="">— اختر —</option>
              <?php foreach ($branches as $bid=>$bname): ?>
                <option value="<?= (int)$bid ?>"><?= htmlspecialchars($bname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">الحالة</label>
            <select class="form-select" name="status" id="f_status">
              <option value="draft">مسودة</option>
              <option value="requested">معتمدة</option>
              <option value="partial">تنفيذ جزئي</option>
              <option value="returned">تم التنفيذ</option>
              <option value="cancelled">ملغي</option>
            </select>
          </div>

          <div class="col-md-3">
            <label class="form-label">تاريخ المرتجع</label>
            <input type="date" class="form-control" name="return_date" id="f_return_date" value="<?= date('Y-m-d') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">عملية شراء (اختياري)</label>
            <input type="number" class="form-control" name="purchase_id" id="f_purchase_id" placeholder="رقم العملية">
          </div>
          <div class="col-md-2">
            <label class="form-label">العملة</label>
            <input type="text" class="form-control" name="currency_code" id="f_currency" value="EGP">
          </div>
          <div class="col-md-2">
            <label class="form-label">خصم عام</label>
            <div class="input-group">
              <select class="form-select" name="discount_type" id="f_discount_type" style="max-width:110px">
                <option value="fixed">قيمة</option>
                <option value="percent">%</option>
              </select>
              <input type="number" step="0.01" class="form-control" name="discount_value" id="f_discount_value" value="0">
            </div>
          </div>
          <div class="col-md-2">
            <label class="form-label">ضريبة عامة %</label>
            <input type="number" step="0.01" class="form-control" name="tax_rate" id="f_tax_rate" value="0">
          </div>
          <div class="col-md-2">
            <label class="form-label">رسوم أخرى</label>
            <input type="number" step="0.01" class="form-control" name="other_fees" id="f_other_fees" value="0">
          </div>
          <div class="col-md-2">
            <label class="form-label">مرفق (إشعار دائن)</label>
            <input type="file" class="form-control" name="attachment" accept="image/*,.pdf">
          </div>

          <div class="col-12">
            <label class="form-label">ملاحظات</label>
            <textarea class="form-control" name="notes" id="f_notes" rows="2"></textarea>
          </div>

          <!-- بنود -->
          <div class="col-12">
            <div class="table-responsive">
              <table class="table table-sm align-middle text-center">
                <thead class="table-light">
                  <tr>
                    <th style="min-width:240px">المتغيّر (SKU)</th>
                    <th>وصف</th>
                    <th class="num">كمية مطلوبة</th>
                    <th class="num">كمية منفّذة</th>
                    <th class="num">تكلفة/وحدة</th>
                    <th class="num">خصم سطر</th>
                    <th class="num">ضريبة %</th>
                    <th class="num">إجمالي سطر</th>
                    <th></th>
                  </tr>
                </thead>
                <tbody id="itemsBody"></tbody>
              </table>
            </div>
            <button type="button" class="btn btn-outline-primary" onclick="addRow()">➕ إضافة بند</button>
          </div>

          <!-- الإجماليات -->
          <div class="col-12">
            <div class="row g-3 justify-content-end">
              <div class="col-md-5">
                <div class="tot-box">
                  <div class="d-flex justify-content-between"><span>المجموع البنود:</span><strong id="t_subtotal">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>خصم عام:</span><strong id="t_hdrdisc">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>ضريبة عامة:</span><strong id="t_tax">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>رسوم أخرى:</span><strong id="t_extra">0.00</strong></div>
                  <hr class="my-2">
                  <div class="d-flex justify-content-between fs-5"><span>الإجمالي (رصيد دائن):</span><strong id="t_total">0.00</strong></div>
                </div>
              </div>
            </div>
          </div>

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          </div>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Modal: Refund -->
<div class="modal fade" id="refundModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-money-bill-transfer me-2"></i> تسجيل استرداد</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="add_refund">
      <input type="hidden" name="id" id="refund_return_id">
      <div class="mb-2">
        <label class="form-label">المبلغ</label>
        <input type="number" step="0.01" class="form-control" name="amount" required>
      </div>
      <div class="mb-2">
        <label class="form-label">الطريقة</label>
        <input type="text" class="form-control" name="method" placeholder="Cash/Bank/Transfer">
      </div>
      <div class="mb-2">
        <label class="form-label">مرجع</label>
        <input type="text" class="form-control" name="ref_no">
      </div>
      <div class="mb-2">
        <label class="form-label">ملاحظات</label>
        <input type="text" class="form-control" name="refund_notes">
      </div>
      <div class="text-center"><button class="btn btn-primary px-4">حفظ</button></div>
    </form>
  </div></div>
</div>

<datalist id="variantsList">
  <?php foreach ($variants as $v): ?>
    <option value="<?= (int)$v['id'] ?> | <?= htmlspecialchars($v['label'], ENT_QUOTES, 'UTF-8') ?>"></option>
  <?php endforeach; ?>
</datalist>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark
  const body = document.body, darkSwitch=document.getElementById('darkSwitch');
  if (localStorage.getItem('darkMode')==='true'){ body.classList.add('dark'); darkSwitch.textContent='☀️'; }
  darkSwitch?.addEventListener('click', ()=>{ body.classList.toggle('dark'); const d=body.classList.contains('dark'); localStorage.setItem('darkMode', d); darkSwitch.textContent = d?'☀️':'🌙'; });

  const returnModal = new bootstrap.Modal(document.getElementById('returnModal'));
  const refundModal = new bootstrap.Modal(document.getElementById('refundModal'));
  const itemsBody = document.getElementById('itemsBody');

  function openCreate(){
    document.getElementById('modalTitle').textContent='مرتجع جديد';
    ['f_id','f_supplier','f_branch','f_purchase_id'].forEach(id=>{ const el=document.getElementById(id); if(el) el.value=''; });
    ['f_currency','f_notes','f_discount_value','f_tax_rate','f_other_fees'].forEach(id=>{ const el=document.getElementById(id); if(el) el.value = (id==='f_currency'?'EGP':'0'); });
    document.getElementById('f_status').value='draft';
    document.getElementById('f_discount_type').value='fixed';
    document.getElementById('f_return_date').value = new Date().toISOString().slice(0,10);
    itemsBody.innerHTML='';
    addRow();
    calcTotals();
  }

  function openEdit(id){
    fetch(location.href + (location.search ? '&':'?') + 'getReturn=' + encodeURIComponent(id))
      .then(r=>r.json())
      .then(data=>{
        if(!data || !data.header){ alert('لم يتم العثور على البيانات'); return; }
        fillForm(data.header, data.items);
        returnModal.show();
      })
      .catch(()=> alert('تعذر التحميل'));
  }

  function fillForm(h, items){
    document.getElementById('modalTitle').textContent='تعديل مرتجع #' + (h.id||'');
    document.getElementById('f_id').value = h.id||'';
    document.getElementById('f_supplier').value = h.supplier_id||'';
    document.getElementById('f_branch').value = h.branch_id||'';
    document.getElementById('f_purchase_id').value = h.purchase_id||'';
    document.getElementById('f_status').value = h.status||'draft';
    document.getElementById('f_return_date').value = h.return_date||'';
    document.getElementById('f_currency').value = h.currency_code||'EGP';
    document.getElementById('f_discount_type').value = h.discount_type||'fixed';
    document.getElementById('f_discount_value').value = h.discount_value||0;
    document.getElementById('f_tax_rate').value = h.tax_rate||0;
    document.getElementById('f_other_fees').value = h.other_fees||0;
    document.getElementById('f_notes').value = h.notes||'';
    itemsBody.innerHTML='';
    (items||[]).forEach(addRowFromItem);
    if ((items||[]).length===0) addRow();
    calcTotals();
  }

  function addRowFromItem(it){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)" value="${it.variant_id} | ${escapeHtml(it.product_name||'')} | ${escapeHtml(it.sku||'')}">
        <input type="hidden" name="item_variant_id[]" value="${it.variant_id}">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control" value="${escapeHtml(it.description||'')}"></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="${it.qty||0}" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="number" name="item_processed_qty[]" class="form-control text-center" value="${it.processed_qty||0}" oninput="clampProcessed(this.closest('tr'));"></td>
      <td><input type="number" step="0.01" name="item_unit_cost[]" class="form-control text-center" value="${it.unit_cost||0}" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="${it.line_discount||0}" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_tax[]" class="form-control text-center" value="${it.line_tax_rate||0}" oninput="calcTotals()"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
  }

  function addRow(){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)">
        <input type="hidden" name="item_variant_id[]">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control"></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="1" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="number" name="item_processed_qty[]" class="form-control text-center" value="0" oninput="clampProcessed(this.closest('tr'));"></td>
      <td><input type="number" step="0.01" name="item_unit_cost[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td><input type="number" step="0.01" name="item_line_tax[]" class="form-control text-center" value="0" oninput="calcTotals()"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
  }

  function clampProcessed(tr){
    const qty = parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
    const proc = tr.querySelector('[name="item_processed_qty[]"]');
    let v = parseFloat(proc.value||0); if (v>qty) v=qty; if (v<0) v=0; proc.value = v;
    calcTotals();
  }

  function parseVariant(inp){
    const hidden = inp.parentElement.querySelector('input[type=hidden]');
    const val = (inp.value||'').trim();
    const id = parseInt(val.split('|')[0],10);
    if (!isNaN(id)) hidden.value = id;
    calcTotals();
  }

  function calcTotals(){
    let subtotal = 0;
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      const q  = parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
      const u  = parseFloat(tr.querySelector('[name="item_unit_cost[]"]').value||0);
      const d  = parseFloat(tr.querySelector('[name="item_line_discount[]"]').value||0);
      const t  = parseFloat(tr.querySelector('[name="item_line_tax[]"]').value||0);
      let line = (q*u) - d; if (line<0) line=0;
      line += line*(t/100);
      subtotal += line;
      tr.querySelector('.line-total').textContent = line.toFixed(2);
    });
    const dtype = document.getElementById('f_discount_type').value;
    const dval  = parseFloat(document.getElementById('f_discount_value').value||0);
    const taxR  = parseFloat(document.getElementById('f_tax_rate').value||0);
    const fees  = parseFloat(document.getElementById('f_other_fees').value||0);

    let hdrDisc = (dtype==='percent') ? (subtotal*(dval/100)) : dval; hdrDisc=Math.max(0,hdrDisc);
    const after = Math.max(0, subtotal - hdrDisc);
    const hdrTax = after*(taxR/100);
    const total = after + hdrTax + fees;

    document.getElementById('t_subtotal').textContent = subtotal.toFixed(2);
    document.getElementById('t_hdrdisc').textContent  = hdrDisc.toFixed(2);
    document.getElementById('t_tax').textContent      = hdrTax.toFixed(2);
    document.getElementById('t_extra').textContent    = fees.toFixed(2);
    document.getElementById('t_total').textContent    = total.toFixed(2);
  }
  ['f_discount_type','f_discount_value','f_tax_rate','f_other_fees'].forEach(id=>{
    const el=document.getElementById(id); el && el.addEventListener('input', calcTotals);
  });

  function openRefund(id){ document.getElementById('refund_return_id').value=id; refundModal.show(); }
  function escapeHtml(s){ return (s||'').toString().replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
</script>
</body>
</html>
