<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = null;

/* ===== CSRF بسيط ===== */
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}
$csrf = $_SESSION['csrf_token'];

/* ===== Helpers ===== */
function normalize_url_or_null(?string $url): ?string {
  $url = trim((string)$url);
  if ($url === '') return null;
  if (!preg_match('~^https?://~i', $url)) $url = 'https://' . $url;
  return $url;
}
function clean_str(?string $s, int $max = 255): ?string {
  $s = trim((string)$s);
  if ($s === '') return null;
  return mb_substr($s, 0, $max);
}
function generate_supplier_code(PDO $pdo): string {
  // SUP-YYYYMMDD-#### (محاولة توليد كود لطيف وفريد)
  for ($i=0; $i<5; $i++) {
    $code = 'SUP-' . date('Ymd') . '-' . str_pad((string)random_int(0, 9999), 4, '0', STR_PAD_LEFT);
    $stmt = $pdo->prepare("SELECT 1 FROM suppliers WHERE SupplierCode = :c LIMIT 1");
    $stmt->execute([':c'=>$code]);
    if (!$stmt->fetch()) return $code;
  }
  return 'SUP-' . time();
}

/* ===== POST Actions ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF). أعد تحميل الصفحة.';
  } else {
    $action = $_POST['action'] ?? '';
    $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;

    try {
      if ($action === 'save') {
        // قراءة الحقول
        $SupplierCode  = clean_str($_POST['SupplierCode'] ?? '', 30);
        $SupplierName  = clean_str($_POST['SupplierName'] ?? '', 150);
        $SupplierEName = clean_str($_POST['SupplierEName'] ?? '', 150);
        $ContactName   = clean_str($_POST['ContactName'] ?? '', 100);
        $Phone         = clean_str($_POST['Phone'] ?? '', 20);
        $Mobile        = clean_str($_POST['Mobile'] ?? '', 20);
        $WhatsApp      = clean_str($_POST['WhatsApp'] ?? '', 20);
        $Email         = clean_str($_POST['Email'] ?? '', 150);
        $Website       = normalize_url_or_null($_POST['Website'] ?? '');

        $Country       = clean_str($_POST['Country'] ?? '', 80);
        $Governate     = clean_str($_POST['Governate'] ?? '', 80);
        $City          = clean_str($_POST['City'] ?? '', 80);
        $Address       = clean_str($_POST['Address'] ?? '', 255);
        $PostalCode    = clean_str($_POST['PostalCode'] ?? '', 20);

        $TaxNumber     = clean_str($_POST['TaxNumber'] ?? '', 50);
        $CommercialReg = clean_str($_POST['CommercialRegister'] ?? '', 50);

        $Currency      = clean_str($_POST['Currency'] ?? 'EGP', 3) ?? 'EGP';
        $PaymentTerms  = (int)($_POST['PaymentTermsDays'] ?? 0);
        $CreditLimit   = (float)($_POST['CreditLimit'] ?? 0);
        $OpeningBal    = (float)($_POST['OpeningBalance'] ?? 0);
        $OpeningType   = ($_POST['OpeningBalanceType'] ?? 'debit') === 'credit' ? 'credit' : 'debit';

        $Notes         = clean_str($_POST['Notes'] ?? '', 2000);
        $IsActive      = isset($_POST['IsActive']) ? 1 : 0;

        if (!$SupplierName) throw new RuntimeException('اسم المورد مطلوب.');

        if ($id > 0) {
          // تحديث
          $sql = "UPDATE suppliers SET
                    SupplierCode = :SupplierCode,
                    SupplierName = :SupplierName,
                    SupplierEName = :SupplierEName,
                    ContactName = :ContactName,
                    Phone = :Phone,
                    Mobile = :Mobile,
                    WhatsApp = :WhatsApp,
                    Email = :Email,
                    Website = :Website,
                    Country = :Country,
                    Governate = :Governate,
                    City = :City,
                    Address = :Address,
                    PostalCode = :PostalCode,
                    TaxNumber = :TaxNumber,
                    CommercialRegister = :CommercialRegister,
                    Currency = :Currency,
                    PaymentTermsDays = :PaymentTermsDays,
                    CreditLimit = :CreditLimit,
                    OpeningBalance = :OpeningBalance,
                    OpeningBalanceType = :OpeningBalanceType,
                    Notes = :Notes,
                    IsActive = :IsActive
                  WHERE id = :id AND DeletedAt IS NULL";
          $stmt = $pdo->prepare($sql);
          $stmt->execute([
            ':SupplierCode'=>$SupplierCode,
            ':SupplierName'=>$SupplierName,
            ':SupplierEName'=>$SupplierEName,
            ':ContactName'=>$ContactName,
            ':Phone'=>$Phone, ':Mobile'=>$Mobile, ':WhatsApp'=>$WhatsApp,
            ':Email'=>$Email, ':Website'=>$Website,
            ':Country'=>$Country, ':Governate'=>$Governate, ':City'=>$City,
            ':Address'=>$Address, ':PostalCode'=>$PostalCode,
            ':TaxNumber'=>$TaxNumber, ':CommercialRegister'=>$CommercialReg,
            ':Currency'=>$Currency, ':PaymentTermsDays'=>$PaymentTerms,
            ':CreditLimit'=>$CreditLimit, ':OpeningBalance'=>$OpeningBal,
            ':OpeningBalanceType'=>$OpeningType, ':Notes'=>$Notes,
            ':IsActive'=>$IsActive, ':id'=>$id
          ]);
          $success = 'تم تحديث المورد بنجاح ✔️';
        } else {
          // إدراج
          if (!$SupplierCode) $SupplierCode = generate_supplier_code($pdo);

          $sql = "INSERT INTO suppliers
            (SupplierCode, SupplierName, SupplierEName, ContactName, Phone, Mobile, WhatsApp, Email, Website,
             Country, Governate, City, Address, PostalCode, TaxNumber, CommercialRegister, Currency, PaymentTermsDays,
             CreditLimit, OpeningBalance, OpeningBalanceType, Notes, IsActive)
            VALUES
            (:SupplierCode,:SupplierName,:SupplierEName,:ContactName,:Phone,:Mobile,:WhatsApp,:Email,:Website,
             :Country,:Governate,:City,:Address,:PostalCode,:TaxNumber,:CommercialRegister,:Currency,:PaymentTermsDays,
             :CreditLimit,:OpeningBalance,:OpeningBalanceType,:Notes,:IsActive)";
          $stmt = $pdo->prepare($sql);
          $stmt->execute([
            ':SupplierCode'=>$SupplierCode, ':SupplierName'=>$SupplierName, ':SupplierEName'=>$SupplierEName,
            ':ContactName'=>$ContactName, ':Phone'=>$Phone, ':Mobile'=>$Mobile, ':WhatsApp'=>$WhatsApp,
            ':Email'=>$Email, ':Website'=>$Website, ':Country'=>$Country, ':Governate'=>$Governate, ':City'=>$City,
            ':Address'=>$Address, ':PostalCode'=>$PostalCode, ':TaxNumber'=>$TaxNumber, ':CommercialRegister'=>$CommercialReg,
            ':Currency'=>$Currency, ':PaymentTermsDays'=>$PaymentTerms, ':CreditLimit'=>$CreditLimit,
            ':OpeningBalance'=>$OpeningBal, ':OpeningBalanceType'=>$OpeningType, ':Notes'=>$Notes, ':IsActive'=>$IsActive
          ]);
          $success = 'تم إضافة المورد بنجاح ✔️';
        }
      } elseif ($action === 'toggle_active' && $id > 0) {
        $pdo->prepare("UPDATE suppliers SET IsActive = 1 - IsActive WHERE id = :id AND DeletedAt IS NULL")
            ->execute([':id'=>$id]);
        $success = 'تم تغيير حالة التفعيل.';
      } elseif ($action === 'soft_delete' && $id > 0) {
        $pdo->prepare("UPDATE suppliers SET IsActive = 0, DeletedAt = NOW() WHERE id = :id AND DeletedAt IS NULL")
            ->execute([':id'=>$id]);
        $success = 'تم أرشفة المورد (حذف منطقي).';
      }
    } catch (Throwable $ex) {
      $error = 'خطأ: ' . htmlspecialchars($ex->getMessage(), ENT_QUOTES, 'UTF-8');
    }
  }
}

/* ===== Search + Pagination ===== */
$q = trim($_GET['q'] ?? '');
$showInactive = isset($_GET['show_inactive']) ? 1 : 0;
$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = 12;
$offset = ($page - 1) * $perPage;

$where = "DeletedAt IS NULL";
$params = [];
if (!$showInactive) { $where .= " AND IsActive = 1"; }
if ($q !== '') {
  $where .= " AND (SupplierName LIKE :q OR SupplierCode LIKE :q OR Mobile LIKE :q OR Phone LIKE :q OR Email LIKE :q)";
  $params[':q'] = "%$q%";
}

$total = (int)$pdo->prepare("SELECT COUNT(*) FROM suppliers WHERE $where")->execute($params) ?: 0;
$stmtCnt = $pdo->prepare("SELECT COUNT(*) AS c FROM suppliers WHERE $where");
$stmtCnt->execute($params);
$total = (int)$stmtCnt->fetchColumn();

$sqlList = "SELECT * FROM suppliers WHERE $where ORDER BY id DESC LIMIT :off, :lim";
$stmt = $pdo->prepare($sqlList);
foreach ($params as $k=>$v) $stmt->bindValue($k, $v, PDO::PARAM_STR);
$stmt->bindValue(':off', $offset, PDO::PARAM_INT);
$stmt->bindValue(':lim', $perPage, PDO::PARAM_INT);
$stmt->execute();
$rows = $stmt->fetchAll();

$totalPages = max(1, (int)ceil($total / $perPage));
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>الموردون</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      /* باستيل ناعم */
      --bg:#f7fafc;
      --text:#1f2937;
      --muted:#6b7280;
      --card:#ffffff;
      --border:rgba(0,0,0,.06);

      --p1:#c9e4de; /* Mint */
      --p2:#f6eac2; /* Sand */
      --p3:#f7d9e3; /* Pink */
      --p4:#d7e3fc; /* Blue */
      --primary:#7cc5b5;      /* زر */
      --primary-600:#5fb3a0;  /* هوفر */
    }
    body.dark{
      --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#234; --p2:#413; --p3:#345; --p4:#264;
      --primary:#4fb79f; --primary-600:#3f9884;
    }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    a{color:#3983c9;text-decoration:none} a:hover{text-decoration:underline}

    header{
      background: linear-gradient(90deg, var(--p1), var(--p2), var(--p3), var(--p4));
      padding:14px 18px; color:#1f2a37; display:flex; align-items:center; gap:12px; justify-content:space-between;
      box-shadow:0 6px 24px rgba(0,0,0,.15);
    }
    .title{font-weight:900; letter-spacing:.2px; text-shadow:0 1px 0 rgba(255,255,255,.5)}
    .user-info{display:flex; align-items:center; gap:10px; background:rgba(255,255,255,.6); padding:6px 10px; border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}

    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .filters{display:flex;gap:10px;align-items:center;flex-wrap:wrap}
    .btn-primary{background:var(--primary);border:none} .btn-primary:hover{background:var(--primary-600)}
    .badge-active{background:#34d399} .badge-inactive{background:#f59e0b}
    .table thead th{color:#6b7280;font-weight:800}
    .table>:not(caption)>*>*{background:transparent}
    .pill{display:inline-block;padding:6px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .pagination .page-link{border-radius:10px;margin:0 3px}
    .back-btn{
  background: rgba(255,255,255,.85);
  color:#1f2937;
  border:0;
  border-radius:999px;
  padding:8px 12px;
  font-weight:800;
  text-decoration:none;
}
.back-btn:hover{ background:#fff; color:#111; text-decoration:none; }

  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= htmlspecialchars($_SESSION['admin_username']) ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>

  <div class="title"><i class="fa-solid fa-truck-field me-2"></i> الموردون</div>

  <!-- زر الرجوع + زر الوضع الليلي -->
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn">
      <i class="fa-solid fa-arrow-right"></i> الرجوع للوحة التحكم
    </a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>


<div class="wrap">
  <div class="card-soft">
    <div class="card-head">
      <div class="filters">
        <form class="d-flex gap-2" method="get">
          <input type="text" class="form-control" style="min-width:260px" name="q" value="<?= htmlspecialchars($q) ?>" placeholder="بحث بالاسم، الكود، الهاتف...">
          <div class="form-check align-self-center">
            <input class="form-check-input" type="checkbox" id="show_inactive" name="show_inactive" value="1" <?= $showInactive ? 'checked':''; ?>>
            <label class="form-check-label" for="show_inactive">إظهار غير المفعلين</label>
          </div>
          <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> بحث</button>
        </form>
      </div>
      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#supplierModal" onclick="openCreate()">
        <i class="fa-solid fa-plus"></i> مورد جديد
      </button>
    </div>

    <div class="p-3">
      <?php if ($success): ?>
        <div class="alert alert-success"><?= $success ?></div>
      <?php elseif ($error): ?>
        <div class="alert alert-danger"><?= $error ?></div>
      <?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle">
          <thead>
            <tr>
              <th>#</th>
              <th>الكود</th>
              <th>الاسم</th>
              <th>مسؤول التواصل</th>
              <th>الهاتف</th>
              <th>الموبايل</th>
              <th>المدينة</th>
              <th>الحالة</th>
              <th class="text-center">إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="9" class="text-center text-muted py-4">لا توجد بيانات.</td></tr>
          <?php else: foreach ($rows as $r): ?>
            <tr data-row='<?= htmlspecialchars(json_encode($r), ENT_QUOTES, "UTF-8") ?>'>
              <td><?= (int)$r['id'] ?></td>
              <td><span class="pill"><?= htmlspecialchars($r['SupplierCode']) ?></span></td>
              <td class="fw-bold"><?= htmlspecialchars($r['SupplierName']) ?></td>
              <td><?= htmlspecialchars($r['ContactName'] ?? '') ?></td>
              <td><?= htmlspecialchars($r['Phone'] ?? '') ?></td>
              <td><?= htmlspecialchars($r['Mobile'] ?? '') ?></td>
              <td><?= htmlspecialchars($r['City'] ?? '') ?></td>
              <td>
                <?php if ((int)$r['IsActive'] === 1): ?>
                  <span class="badge badge-active">مفعل</span>
                <?php else: ?>
                  <span class="badge badge-inactive">غير مفعل</span>
                <?php endif; ?>
              </td>
              <td class="text-center">
                <div class="btn-group">
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(this)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <form method="post" class="d-inline" onsubmit="return confirm('تأكيد تغيير الحالة؟')">
                    <input type="hidden" name="csrf" value="<?= $csrf ?>">
                    <input type="hidden" name="action" value="toggle_active">
                    <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                    <button class="btn btn-sm btn-outline-warning"><i class="fa-solid fa-toggle-on"></i></button>
                  </form>
                  <form method="post" class="d-inline" onsubmit="return confirm('أرشفة المورد (حذف منطقي)؟')">
                    <input type="hidden" name="csrf" value="<?= $csrf ?>">
                    <input type="hidden" name="action" value="soft_delete">
                    <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                    <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                  </form>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>

      <!-- Pagination -->
      <?php if ($totalPages > 1): ?>
        <nav class="d-flex justify-content-center">
          <ul class="pagination">
            <?php for ($p=1; $p <= $totalPages; $p++): ?>
              <?php
                $qs = $_GET; $qs['page'] = $p;
                $url = '?' . http_build_query($qs);
              ?>
              <li class="page-item <?= $p === $page ? 'active':'' ?>">
                <a class="page-link" href="<?= htmlspecialchars($url) ?>"><?= $p ?></a>
              </li>
            <?php endfor; ?>
          </ul>
        </nav>
      <?php endif; ?>

    </div>
  </div>
</div>

<!-- Modal: Create/Edit -->
<div class="modal fade" id="supplierModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-truck-field me-2"></i> <span id="modalTitle">مورد جديد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <form method="post" class="modal-body row g-3">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save">
        <input type="hidden" name="id" id="f_id" value="0">

        <div class="col-md-4">
          <label class="form-label">الكود</label>
          <input type="text" class="form-control" name="SupplierCode" id="f_code" placeholder="سيتولد تلقائيًا إن تركته فارغًا">
        </div>
        <div class="col-md-4">
          <label class="form-label">الاسم (عربي) *</label>
          <input type="text" class="form-control" name="SupplierName" id="f_name" required>
        </div>
        <div class="col-md-4">
          <label class="form-label">الاسم (إنجليزي)</label>
          <input type="text" class="form-control" name="SupplierEName" id="f_name_en">
        </div>

        <div class="col-md-4">
          <label class="form-label">مسؤول التواصل</label>
          <input type="text" class="form-control" name="ContactName" id="f_contact">
        </div>
        <div class="col-md-4">
          <label class="form-label">الهاتف</label>
          <input type="text" class="form-control" name="Phone" id="f_phone">
        </div>
        <div class="col-md-4">
          <label class="form-label">الموبايل</label>
          <input type="text" class="form-control" name="Mobile" id="f_mobile">
        </div>

        <div class="col-md-4">
          <label class="form-label">واتساب</label>
          <input type="text" class="form-control" name="WhatsApp" id="f_whatsapp">
        </div>
        <div class="col-md-4">
          <label class="form-label">البريد الإلكتروني</label>
          <input type="email" class="form-control" name="Email" id="f_email">
        </div>
        <div class="col-md-4">
          <label class="form-label">الموقع الإلكتروني</label>
          <input type="text" class="form-control" name="Website" id="f_website" placeholder="example.com">
        </div>

        <div class="col-md-3">
          <label class="form-label">الدولة</label>
          <input type="text" class="form-control" name="Country" id="f_country">
        </div>
        <div class="col-md-3">
          <label class="form-label">المحافظة</label>
          <input type="text" class="form-control" name="Governate" id="f_governate">
        </div>
        <div class="col-md-3">
          <label class="form-label">المدينة</label>
          <input type="text" class="form-control" name="City" id="f_city">
        </div>
        <div class="col-md-3">
          <label class="form-label">الرمز البريدي</label>
          <input type="text" class="form-control" name="PostalCode" id="f_postal">
        </div>

        <div class="col-12">
          <label class="form-label">العنوان</label>
          <input type="text" class="form-control" name="Address" id="f_address">
        </div>

        <div class="col-md-3">
          <label class="form-label">رقم ضريبي</label>
          <input type="text" class="form-control" name="TaxNumber" id="f_tax">
        </div>
        <div class="col-md-3">
          <label class="form-label">سجل تجاري</label>
          <input type="text" class="form-control" name="CommercialRegister" id="f_cr">
        </div>
        <div class="col-md-2">
          <label class="form-label">العملة</label>
          <input type="text" class="form-control" name="Currency" id="f_currency" value="EGP">
        </div>
        <div class="col-md-2">
          <label class="form-label">شروط الدفع (يوم)</label>
          <input type="number" min="0" class="form-control" name="PaymentTermsDays" id="f_terms" value="0">
        </div>
        <div class="col-md-2">
          <label class="form-label">حد ائتماني</label>
          <input type="number" step="0.01" min="0" class="form-control" name="CreditLimit" id="f_credit" value="0">
        </div>

        <div class="col-md-3">
          <label class="form-label">رصيد افتتاحي</label>
          <input type="number" step="0.01" class="form-control" name="OpeningBalance" id="f_opening" value="0">
        </div>
        <div class="col-md-3">
          <label class="form-label">نوع الرصيد</label>
          <select class="form-select" name="OpeningBalanceType" id="f_opening_type">
            <option value="debit">مدين</option>
            <option value="credit">دائن</option>
          </select>
        </div>

        <div class="col-12">
          <label class="form-label">ملاحظات</label>
          <textarea class="form-control" name="Notes" id="f_notes" rows="2"></textarea>
        </div>

        <div class="col-12">
          <div class="form-check">
            <input class="form-check-input" type="checkbox" id="f_active" name="IsActive" checked>
            <label class="form-check-label" for="f_active">مفعّل</label>
          </div>
        </div>

        <div class="col-12 text-center mt-2">
          <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          <button type="button" class="btn btn-outline-secondary ms-2" data-bs-dismiss="modal">إلغاء</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark mode
  const body = document.body;
  const darkSwitch = document.getElementById("darkSwitch");
  if (localStorage.getItem("darkMode") === "true") { body.classList.add("dark"); darkSwitch.textContent = "☀️"; }
  darkSwitch.addEventListener("click", ()=> {
    body.classList.toggle("dark");
    const isDark = body.classList.contains("dark");
    localStorage.setItem("darkMode", isDark);
    darkSwitch.textContent = isDark ? "☀️" : "🌙";
  });

  const supplierModal = new bootstrap.Modal(document.getElementById('supplierModal'));
  const modalTitle = document.getElementById('modalTitle');

  function fillForm(data){
    document.getElementById('f_id').value         = data?.id || 0;
    document.getElementById('f_code').value       = data?.SupplierCode || '';
    document.getElementById('f_name').value       = data?.SupplierName || '';
    document.getElementById('f_name_en').value    = data?.SupplierEName || '';
    document.getElementById('f_contact').value    = data?.ContactName || '';
    document.getElementById('f_phone').value      = data?.Phone || '';
    document.getElementById('f_mobile').value     = data?.Mobile || '';
    document.getElementById('f_whatsapp').value   = data?.WhatsApp || '';
    document.getElementById('f_email').value      = data?.Email || '';
    document.getElementById('f_website').value    = data?.Website || '';
    document.getElementById('f_country').value    = data?.Country || '';
    document.getElementById('f_governate').value  = data?.Governate || '';
    document.getElementById('f_city').value       = data?.City || '';
    document.getElementById('f_postal').value     = data?.PostalCode || '';
    document.getElementById('f_address').value    = data?.Address || '';
    document.getElementById('f_tax').value        = data?.TaxNumber || '';
    document.getElementById('f_cr').value         = data?.CommercialRegister || '';
    document.getElementById('f_currency').value   = data?.Currency || 'EGP';
    document.getElementById('f_terms').value      = data?.PaymentTermsDays ?? 0;
    document.getElementById('f_credit').value     = data?.CreditLimit ?? 0;
    document.getElementById('f_opening').value    = data?.OpeningBalance ?? 0;
    document.getElementById('f_opening_type').value = data?.OpeningBalanceType || 'debit';
    document.getElementById('f_notes').value      = data?.Notes || '';
    document.getElementById('f_active').checked   = (parseInt(data?.IsActive ?? 1) === 1);
  }

  function openCreate(){
    modalTitle.textContent = 'مورد جديد';
    fillForm(null);
  }

  function openEdit(btn){
    const tr = btn.closest('tr');
    const data = JSON.parse(tr.getAttribute('data-row'));
    modalTitle.textContent = 'تعديل مورد';
    fillForm(data);
    supplierModal.show();
  }
</script>
</body>
</html>
