<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ====== صلاحيات ====== */
$canEdit = true;

/* ====== CSRF ====== */
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf_token'];

/* ====== Helpers ====== */
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function hasColumn(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema=DATABASE() AND table_name=? AND column_name=?");
  $st->execute([$table,$col]); return (int)$st->fetchColumn() > 0;
}
function clamp_int($v,$min,$max){ $v=(int)$v; return max($min, min($max,$v)); }
function money($n){ return number_format((float)$n,2); }
function get_param($k,$d=null){ return isset($_GET[$k]) ? $_GET[$k] : $d; }

/* ====== فروع ====== */
try {
  $branches = $pdo->query("SELECT BranchID, BranchName FROM Branches ORDER BY BranchName")->fetchAll(PDO::FETCH_KEY_PAIR);
} catch (Throwable $e){ $branches = []; }

/* ====== متغيّرات للبحث السريع (دatalist) ====== */
$variantsList = $pdo->query("
  SELECT v.id, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-'),' | ',v.sku) AS label
  FROM product_variants v
  JOIN products p ON p.id=v.product_id
  WHERE v.deleted_at IS NULL
  ORDER BY p.name_ar, v.id
  LIMIT 2000
")->fetchAll(PDO::FETCH_ASSOC);

/* ====== آخر تكلفة شراء لعنصر ====== */
function last_purchase_cost(PDO $pdo, int $variantId): float {
  try {
    $st=$pdo->prepare("SELECT unit_cost FROM purchase_items WHERE variant_id=? ORDER BY id DESC LIMIT 1");
    $st->execute([$variantId]);
    $x = $st->fetchColumn();
    return $x!==false ? (float)$x : 0.0;
  } catch(Throwable $e){ return 0.0; }
}

/* ====== Endpoints (قبل أي HTML) ====== */
// JSON: سجل حركات لمتغيّر (اختياري حسب فرع)
if (isset($_GET['movements'])) {
  header('Content-Type: application/json; charset=utf-8');
  $vid = (int)($_GET['variant_id'] ?? 0);
  $bid = ($_GET['branch_id'] ?? '')!=='' ? (int)$_GET['branch_id'] : null;
  if ($vid<=0){ echo json_encode(['ok'=>false,'error'=>'variant_id required']); exit; }
  $sql = "SELECT id, created_at, branch_id, qty_change, unit_cost, ref_type, ref_id, notes
          FROM inventory_movements WHERE variant_id=?".($bid? " AND branch_id=?": "")." ORDER BY id DESC LIMIT 200";
  $st = $pdo->prepare($sql);
  $st->execute($bid? [$vid,$bid] : [$vid]);
  echo json_encode(['ok'=>true,'rows'=>$st->fetchAll(PDO::FETCH_ASSOC)], JSON_UNESCAPED_UNICODE);
  exit;
}

// CSV: تصدير النتائج الحالية
if (isset($_GET['export']) && $_GET['export']=='1') {
  $mode      = get_param('mode','by_branch');
  $q         = trim(get_param('q',''));
  $fBranch   = get_param('branch_id', null);
  $only_low  = isset($_GET['only_low']);
  $only_zero = isset($_GET['only_zero']);
  $only_neg  = isset($_GET['only_neg']);
  [$rows,] = fetch_inventory_rows($pdo, $mode, $q, $fBranch, $only_low, $only_zero, $only_neg, 'name', 1, 50000);
  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename="inventory_export.csv"');
  $out = fopen('php://output','w');
  fputcsv($out, $mode==='by_variant'
    ? ['VariantID','SKU','Name AR','Name EN','TotalQty','TotalMin','Shortage']
    : ['VariantID','SKU','Name AR','Name EN','BranchID','BranchName','Qty','MinQty','Shortage']
  );
  foreach ($rows as $r){
    if ($mode==='by_variant'){
      fputcsv($out, [$r['variant_id'],$r['sku'],$r['name_ar'],$r['name_en'],(int)$r['qty'],(int)$r['min_qty'], max(0,(int)$r['min_qty']-(int)$r['qty'])]);
    } else {
      fputcsv($out, [$r['variant_id'],$r['sku'],$r['name_ar'],$r['name_en'],$r['branch_id'],$r['BranchName'],(int)$r['quantity'],(int)$r['min_quantity'], max(0,(int)$r['min_quantity']-(int)$r['quantity'])]);
    }
  }
  fclose($out);
  exit;
}

/* ====== POST Actions ====== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error='طلب غير صالح (CSRF).';
  } elseif (!$canEdit) {
    $error='لا تملك صلاحية.';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action==='save_prefs'){
        $prefs = [
          'default_branch' => ($_POST['default_branch'] ?? '')!=='' ? (int)$_POST['default_branch'] : null,
          'default_mode'   => in_array($_POST['default_mode'] ?? 'by_branch',['by_branch','by_variant'],true) ? $_POST['default_mode'] : 'by_branch',
          'only_low'       => isset($_POST['only_low']),
          'per_page'       => clamp_int($_POST['per_page'] ?? 50, 10, 200)
        ];
        $_SESSION['inv_prefs'] = $prefs;
        setcookie('inv_prefs', json_encode($prefs, JSON_UNESCAPED_UNICODE), time()+60*60*24*30, '/');
        $success='✔️ تم حفظ التفضيلات';

      } elseif ($action==='update_min') {
        $variant_id = (int)($_POST['variant_id'] ?? 0);
        $branch_id  = (int)($_POST['branch_id'] ?? 0);
        $min_qty    = max(0,(int)($_POST['min_qty'] ?? 0));
        if ($variant_id<=0 || $branch_id<=0) throw new RuntimeException('بيانات غير كافية.');
        $pdo->prepare("
          INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
          VALUES (?,?,0,?)
          ON DUPLICATE KEY UPDATE min_quantity=VALUES(min_quantity)
        ")->execute([$variant_id,$branch_id,$min_qty]);
        $success='✔️ تم تحديث الحد الأدنى';

      } elseif ($action==='transfer'){
        $variant_id = (int)($_POST['variant_id'] ?? 0);
        $from_id    = (int)($_POST['from_branch'] ?? 0);
        $to_id      = (int)($_POST['to_branch'] ?? 0);
        $qty        = max(1,(int)($_POST['qty'] ?? 0));
        if ($variant_id<=0 || $from_id<=0 || $to_id<=0 || $from_id===$to_id) throw new RuntimeException('تحويل غير صالح.');
        $pdo->beginTransaction();
        // تحقق المتاح
        $st=$pdo->prepare("SELECT quantity FROM product_variant_stock WHERE variant_id=? AND branch_id=? FOR UPDATE");
        $st->execute([$variant_id,$from_id]);
        $fromQty=(int)($st->fetchColumn() ?: 0);
        if ($fromQty < $qty) throw new RuntimeException('المخزون غير كافٍ بالفرع المصدر.');
        // خصم واضافة
        $pdo->prepare("UPDATE product_variant_stock SET quantity=GREATEST(0,quantity-?) WHERE variant_id=? AND branch_id=?")
            ->execute([$qty,$variant_id,$from_id]);
        $pdo->prepare("
          INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
          VALUES (?,?,?,0) ON DUPLICATE KEY UPDATE quantity=quantity+VALUES(quantity)
        ")->execute([$variant_id,$to_id,$qty]);
        // حركات
        $uc = last_purchase_cost($pdo,$variant_id);
        $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                       VALUES (?,?,?,?, 'transfer_out', NULL)")
            ->execute([$variant_id,$from_id,-$qty,$uc]);
        $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                       VALUES (?,?,?,?, 'transfer_in', NULL)")
            ->execute([$variant_id,$to_id,$qty,$uc]);
        $pdo->commit();
        $success='✔️ تم التحويل';

      } elseif ($action==='adjust'){
        $variant_id = (int)($_POST['variant_id'] ?? 0);
        $branch_id  = (int)($_POST['branch_id'] ?? 0);
        $delta      = (int)($_POST['delta'] ?? 0);
        $reason     = trim((string)($_POST['reason'] ?? 'manual_adjust'));
        if ($variant_id<=0 || $branch_id<=0 || $delta===0) throw new RuntimeException('قيمة ضبط غير صالحة.');
        $pdo->beginTransaction();
        // قراءة & قفل
        $st=$pdo->prepare("SELECT quantity FROM product_variant_stock WHERE variant_id=? AND branch_id=? FOR UPDATE");
        $st->execute([$variant_id,$branch_id]);
        $q = (int)($st->fetchColumn() ?: 0);
        $new = $q + $delta;
        if ($new < 0) throw new RuntimeException('العملية ستنتج سالب.');
        if ($st->rowCount()===0){
          $pdo->prepare("INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity) VALUES (?,?,?,0)")
              ->execute([$variant_id,$branch_id,$new]);
        } else {
          $pdo->prepare("UPDATE product_variant_stock SET quantity=? WHERE variant_id=? AND branch_id=?")
              ->execute([$new,$variant_id,$branch_id]);
        }
        $uc = last_purchase_cost($pdo,$variant_id);
        $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id, notes)
                       VALUES (?,?,?,?, 'manual_adjust', NULL, ?)")
            ->execute([$variant_id,$branch_id,$delta,$uc,$reason]);
        $pdo->commit();
        $success='✔️ تم ضبط المخزون';
      }
    } catch(Throwable $e){
      if ($pdo->inTransaction()) $pdo->rollBack();
      $error = 'خطأ: '.esc($e->getMessage());
    }
  }
}

/* ====== “تفضيلاتي” ====== */
if (!isset($_SESSION['inv_prefs']) && !empty($_COOKIE['inv_prefs'])) {
  $tmp = json_decode($_COOKIE['inv_prefs'], true);
  if (is_array($tmp)) $_SESSION['inv_prefs'] = $tmp;
}
$_SESSION['inv_prefs'] = $_SESSION['inv_prefs'] ?? [
  'default_branch'=> null,
  'default_mode'  => 'by_branch',
  'only_low'      => false,
  'per_page'      => 50
];
$pref = $_SESSION['inv_prefs'];

/* ====== بارامترات العرض ====== */
$q           = trim(get_param('q',''));
$mode        = get_param('mode', $pref['default_mode']); // by_branch | by_variant
$fBranch     = get_param('branch_id', $pref['default_branch']);
$only_low    = isset($_GET['only_low']) ? true : (bool)$pref['only_low'];
$only_zero   = isset($_GET['only_zero']);
$only_neg    = isset($_GET['only_neg']);
$sort        = get_param('sort','qty_asc'); // qty_asc|qty_desc|name|sku|shortage
$page        = clamp_int(get_param('page',1),1,100000);
$per         = clamp_int(get_param('per', $pref['per_page']), 10, 200);

/* ====== دالة جلب النتائج (مع إجمالي للترقيم) ====== */
function fetch_inventory_rows(PDO $pdo, string $mode, string $q, $fBranch, bool $only_low, bool $only_zero, bool $only_neg, string $sort, int $page, int $per): array {
  $params = [];
  $where  = [];

  if ($q !== '') {
    $where[] = "(p.name_ar LIKE :q OR p.name_en LIKE :q OR v.sku LIKE :q)";
    $params[':q'] = "%$q%";
  }

  if ($mode === 'by_branch') {
    if ($fBranch) { $where[]="s.branch_id=:bid"; $params[':bid']=(int)$fBranch; }
    $whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';
    $base = "
      SELECT s.branch_id, b.BranchName, v.id AS variant_id, v.sku, p.name_ar, p.name_en,
             COALESCE(s.quantity,0) AS quantity, COALESCE(s.min_quantity,0) AS min_quantity,
             GREATEST(COALESCE(s.min_quantity,0) - COALESCE(s.quantity,0), 0) AS shortage
      FROM product_variant_stock s
      JOIN product_variants v ON v.id=s.variant_id
      JOIN products p ON p.id=v.product_id
      JOIN Branches b ON b.BranchID=s.branch_id
      $whereSql
    ";
    $having = [];
    if ($only_low)  $having[] = "shortage > 0";
    if ($only_zero) $having[] = "quantity = 0";
    if ($only_neg)  $having[] = "quantity < 0";
    $havingSql = $having ? ('HAVING '.implode(' AND ',$having)) : '';

    $order = match($sort){
      'qty_desc' => 'quantity DESC',
      'name'     => 'p.name_ar ASC, v.sku ASC',
      'sku'      => 'v.sku ASC',
      'shortage' => 'shortage DESC',
      default    => 'quantity ASC'
    };

    // إجمالي
    $cnt = $pdo->prepare("SELECT COUNT(*) FROM ($base) t $havingSql");
    $cnt->execute($params);
    $total = (int)$cnt->fetchColumn();

    $offset = ($page-1)*$per;
    $sql = "SELECT * FROM ($base) t $havingSql ORDER BY $order LIMIT $per OFFSET $offset";
    $st = $pdo->prepare($sql);
    $st->execute($params);
    return [$st->fetchAll(PDO::FETCH_ASSOC), $total];

  } else { // by_variant (تجميع)
    $whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';
    $base = "
      SELECT v.id AS variant_id, v.sku, p.name_ar, p.name_en,
             COALESCE(SUM(s.quantity),0) AS qty,
             COALESCE(SUM(s.min_quantity),0) AS min_qty,
             GREATEST(COALESCE(SUM(s.min_quantity),0) - COALESCE(SUM(s.quantity),0), 0) AS shortage
      FROM product_variants v
      JOIN products p ON p.id=v.product_id
      LEFT JOIN product_variant_stock s ON s.variant_id=v.id
      $whereSql
      GROUP BY v.id
    ";
    $having = [];
    if ($fBranch){ // فلترة حسب فرع بعرض تجميعي: نأخذ فقط المتغيّرات التي لها صف في هذا الفرع
      $having[] = "MAX(CASE WHEN s.branch_id = ".(int)$fBranch." THEN 1 ELSE 0 END) = 1";
    }
    if ($only_low)  $having[] = "shortage > 0";
    if ($only_zero) $having[] = "qty = 0";
    if ($only_neg)  $having[] = "qty < 0";
    $havingSql = $having ? ('HAVING '.implode(' AND ',$having)) : '';

    $order = match($sort){
      'qty_desc' => 'qty DESC',
      'name'     => 'p.name_ar ASC, v.sku ASC',
      'sku'      => 'v.sku ASC',
      'shortage' => 'shortage DESC',
      default    => 'qty ASC'
    };

    $cnt = $pdo->prepare("SELECT COUNT(*) FROM ($base) t $havingSql");
    $cnt->execute($params);
    $total = (int)$cnt->fetchColumn();

    $offset = ($page-1)*$per;
    $sql = "SELECT * FROM ($base) t $havingSql ORDER BY $order LIMIT $per OFFSET $offset";
    $st = $pdo->prepare($sql);
    $st->execute($params);
    return [$st->fetchAll(PDO::FETCH_ASSOC), $total];
  }
}

/* ====== جلب البيانات ====== */
[$rows, $total] = fetch_inventory_rows($pdo, $mode, $q, $fBranch, $only_low, $only_zero, $only_neg, $sort, $page, $per);
$total_pages = max(1, (int)ceil($total / $per));

/* ====== اقتراحات إعادة الطلب (أول 20 منخفض) ====== */
$reorder = [];
if ($mode==='by_branch'){
  $reorder = array_values(array_filter($rows, fn($r)=> (int)$r['min_quantity'] > (int)$r['quantity']));
  usort($reorder, fn($a,$b)=> ($b['shortage']<=>$a['shortage']));
  $reorder = array_slice($reorder,0,20);
} else {
  $reorder = array_values(array_filter($rows, fn($r)=> (int)$r['min_qty'] > (int)$r['qty']));
  usort($reorder, fn($a,$b)=> ($b['shortage']<=>$a['shortage']));
  $reorder = array_slice($reorder,0,20);
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>المخزون</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#fff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{
      background:linear-gradient(90deg,var(--p1),var(--p2),var(--p3),var(--p4));
      padding:14px 18px; display:flex; align-items:center; justify-content:space-between; box-shadow:0 6px 24px rgba(0,0,0,.15);
    }
    .user-info{display:flex;align-items:center;gap:10px;background:rgba(255,255,255,.6);padding:6px 10px;border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}
    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .muted{color:var(--muted)}
    .table>:not(caption)>*>*{background:transparent}
    .num{min-width:90px;text-align:center}
    .soft{background:rgba(0,0,0,.03);border-radius:10px;padding:6px 10px}
    .low{background:#fff3cd;border:1px solid #ffe69c;border-radius:999px;padding:2px 8px}
    .neg{background:#f8d7da;border:1px solid #f5c2c7;border-radius:999px;padding:2px 8px}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= esc($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= esc($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-warehouse me-2"></i> المخزون</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">

  <!-- شريط الفلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:260px" placeholder="بحث بالاسم/الـSKU" value="<?= esc($q) ?>">

        <select class="form-select" name="mode" style="min-width:180px">
          <option value="by_branch" <?= $mode==='by_branch'?'selected':''; ?>>حسب الفرع</option>
          <option value="by_variant" <?= $mode==='by_variant'?'selected':''; ?>>تجميعي لكل متغيّر</option>
        </select>

        <select class="form-select" name="branch_id" style="min-width:200px">
          <option value="">كل الفروع</option>
          <?php foreach ($branches as $bid=>$bn): ?>
            <option value="<?= (int)$bid ?>" <?= (string)$fBranch===(string)$bid?'selected':''; ?>><?= esc($bn) ?></option>
          <?php endforeach; ?>
        </select>

        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="only_low" name="only_low" value="1" <?= $only_low?'checked':''; ?>>
          <label class="form-check-label" for="only_low">المنخفض فقط</label>
        </div>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="only_zero" name="only_zero" value="1" <?= $only_zero?'checked':''; ?>>
          <label class="form-check-label" for="only_zero">المنتهي = 0</label>
        </div>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="only_neg" name="only_neg" value="1" <?= $only_neg?'checked':''; ?>>
          <label class="form-check-label" for="only_neg">السالب</label>
        </div>

        <select class="form-select" name="sort" style="min-width:180px">
          <option value="qty_asc"  <?= $sort==='qty_asc'?'selected':''; ?>>الكمية (تصاعدي)</option>
          <option value="qty_desc" <?= $sort==='qty_desc'?'selected':''; ?>>الكمية (تنازلي)</option>
          <option value="shortage" <?= $sort==='shortage'?'selected':''; ?>>العجز</option>
          <option value="name"     <?= $sort==='name'?'selected':''; ?>>الاسم</option>
          <option value="sku"      <?= $sort==='sku'?'selected':''; ?>>SKU</option>
        </select>

        <select class="form-select" name="per" style="min-width:120px">
          <?php foreach ([25,50,100,150,200] as $n): ?>
            <option value="<?= $n ?>" <?= $per===$n?'selected':''; ?>><?= $n ?>/صفحة</option>
          <?php endforeach; ?>
        </select>

        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $fBranch || $only_low || $only_zero || $only_neg || $mode!=='by_branch' || $sort!=='qty_asc' || (int)$per!== (int)$pref['per_page']): ?>
          <a href="inventory.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>

        <a class="btn btn-outline-success" href="?<?= http_build_query(array_merge($_GET,['export'=>1])) ?>">
          <i class="fa-solid fa-file-csv"></i> تصدير CSV
        </a>
      </form>

      <div class="d-flex flex-wrap gap-2">
        <button class="btn btn-outline-secondary" data-bs-toggle="modal" data-bs-target="#prefsModal"><i class="fa-solid fa-sliders"></i> تفضيلاتي</button>
        <?php if ($canEdit): ?>
          <button class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#minModal"><i class="fa-solid fa-gauge-simple-high"></i> حدّ أدنى</button>
          <button class="btn btn-outline-warning" data-bs-toggle="modal" data-bs-target="#transferModal"><i class="fa-solid fa-right-left"></i> تحويل</button>
          <button class="btn btn-outline-dark" data-bs-toggle="modal" data-bs-target="#adjustModal"><i class="fa-solid fa-scale-balanced"></i> تسوية</button>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <!-- النتائج -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
          <?php if ($mode==='by_branch'): ?>
            <tr>
              <th>#</th>
              <th>SKU</th>
              <th>الاسم (AR / EN)</th>
              <th>الفرع</th>
              <th>الكمية</th>
              <th>الحد الأدنى</th>
              <th>العجز</th>
              <th>إجراءات</th>
            </tr>
          <?php else: ?>
            <tr>
              <th>#</th>
              <th>SKU</th>
              <th>الاسم (AR / EN)</th>
              <th>إجمالي الكمية</th>
              <th>إجمالي الحد الأدنى</th>
              <th>العجز</th>
              <th>إجراءات</th>
            </tr>
          <?php endif; ?>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="<?= $mode==='by_branch'? 8:7 ?>" class="text-muted">لا توجد بيانات.</td></tr>
          <?php else: ?>
            <?php foreach ($rows as $i=>$r): ?>
              <?php if ($mode==='by_branch'):
                $short = max(0, (int)$r['min_quantity'] - (int)$r['quantity']);
                $neg = (int)$r['quantity'] < 0;
              ?>
              <tr>
                <td><?= (int)$r['variant_id'] ?></td>
                <td><?= esc($r['sku']) ?></td>
                <td class="text-start"><?= esc($r['name_ar']) ?><div class="muted small"><?= esc($r['name_en']) ?></div></td>
                <td><span class="pill"><?= esc($r['BranchName']) ?></span></td>
                <td><?= (int)$r['quantity'] ?><?php if($neg): ?> <span class="neg">سالب</span><?php endif; ?></td>
                <td><?= (int)$r['min_quantity'] ?></td>
                <td><?= $short ?></td>
                <td>
                  <div class="btn-group">
                    <button class="btn btn-sm btn-outline-secondary" onclick="openMov(<?= (int)$r['variant_id'] ?>, <?= (int)$r['branch_id'] ?>)"><i class="fa-solid fa-clock-rotate-left"></i></button>
                    <?php if ($canEdit): ?>
                      <button class="btn btn-sm btn-outline-primary" onclick="prefillMin(<?= (int)$r['variant_id'] ?>, <?= (int)$r['branch_id'] ?>, <?= (int)$r['min_quantity'] ?>)" data-bs-toggle="modal" data-bs-target="#minModal"><i class="fa-solid fa-gauge-high"></i></button>
                      <button class="btn btn-sm btn-outline-warning" onclick="prefillTransfer(<?= (int)$r['variant_id'] ?>, <?= (int)$r['branch_id'] ?>)" data-bs-toggle="modal" data-bs-target="#transferModal"><i class="fa-solid fa-right-left"></i></button>
                      <button class="btn btn-sm btn-outline-dark" onclick="prefillAdjust(<?= (int)$r['variant_id'] ?>, <?= (int)$r['branch_id'] ?>)" data-bs-toggle="modal" data-bs-target="#adjustModal"><i class="fa-solid fa-scale-balanced"></i></button>
                    <?php endif; ?>
                  </div>
                </td>
              </tr>
              <?php else:
                $short = max(0, (int)$r['min_qty'] - (int)$r['qty']);
                $neg = (int)$r['qty'] < 0;
              ?>
              <tr>
                <td><?= (int)$r['variant_id'] ?></td>
                <td><?= esc($r['sku']) ?></td>
                <td class="text-start"><?= esc($r['name_ar']) ?><div class="muted small"><?= esc($r['name_en']) ?></div></td>
                <td><?= (int)$r['qty'] ?><?php if($neg): ?> <span class="neg">سالب</span><?php endif; ?></td>
                <td><?= (int)$r['min_qty'] ?></td>
                <td><?= $short ?></td>
                <td>
                  <div class="btn-group">
                    <button class="btn btn-sm btn-outline-secondary" onclick="openMov(<?= (int)$r['variant_id'] ?>, '')"><i class="fa-solid fa-clock-rotate-left"></i></button>
                    <?php if ($canEdit): ?>
                      <button class="btn btn-sm btn-outline-primary" onclick="prefillMin(<?= (int)$r['variant_id'] ?>, <?= (int)($fBranch ?: 0) ?>, 0)" data-bs-toggle="modal" data-bs-target="#minModal" title="تحديد حد أدنى لفرع معيّن"><i class="fa-solid fa-gauge-high"></i></button>
                      <button class="btn btn-sm btn-outline-warning" onclick="prefillTransfer(<?= (int)$r['variant_id'] ?>, <?= (int)($fBranch ?: 0) ?>)" data-bs-toggle="modal" data-bs-target="#transferModal"><i class="fa-solid fa-right-left"></i></button>
                      <button class="btn btn-sm btn-outline-dark" onclick="prefillAdjust(<?= (int)$r['variant_id'] ?>, <?= (int)($fBranch ?: 0) ?>)" data-bs-toggle="modal" data-bs-target="#adjustModal"><i class="fa-solid fa-scale-balanced"></i></button>
                    <?php endif; ?>
                  </div>
                </td>
              </tr>
              <?php endif; ?>
            <?php endforeach; ?>
          <?php endif; ?>
          </tbody>
        </table>
      </div>

      <!-- ترقيم -->
      <?php if ($total_pages>1): ?>
      <nav class="mt-2">
        <ul class="pagination justify-content-center">
          <?php
            $base = $_GET; unset($base['page']);
            $mk = function($p) use($base){ $base['page']=$p; return '?'.http_build_query($base); };
            $start = max(1,$page-2); $end=min($total_pages,$page+2);
          ?>
          <li class="page-item <?= $page<=1?'disabled':'' ?>"><a class="page-link" href="<?= $mk($page-1) ?>">&laquo;</a></li>
          <?php for($p=$start;$p<=$end;$p++): ?>
            <li class="page-item <?= $p===$page?'active':'' ?>"><a class="page-link" href="<?= $mk($p) ?>"><?= $p ?></a></li>
          <?php endfor; ?>
          <li class="page-item <?= $page>=$total_pages?'disabled':'' ?>"><a class="page-link" href="<?= $mk($page+1) ?>">&raquo;</a></li>
        </ul>
      </nav>
      <?php endif; ?>
    </div>
  </div>

  <!-- اقتراحات إعادة الطلب -->
  <div class="card-soft mt-3">
    <div class="card-head">
      <div class="fw-bold"><i class="fa-regular fa-lightbulb me-2"></i> اقتراحات إعادة الطلب</div>
      <div class="muted small">يعتمد على الحد الأدنى مقابل الكمية الحالية.</div>
    </div>
    <div class="p-3">
      <?php if (!$reorder): ?>
        <div class="text-muted">لا توجد عناصر منخفضة حاليًا.</div>
      <?php else: ?>
        <div class="table-responsive">
          <table class="table table-sm align-middle text-center">
            <thead class="table-light">
              <tr>
                <th>#</th><th>SKU</th><th>الاسم</th>
                <?php if ($mode==='by_branch'): ?><th>الفرع</th><?php endif; ?>
                <th>المتاح</th><th>الحد الأدنى</th><th>العجز المقترح</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($reorder as $rr): ?>
                <?php if ($mode==='by_branch'): ?>
                  <tr>
                    <td><?= (int)$rr['variant_id'] ?></td>
                    <td><?= esc($rr['sku']) ?></td>
                    <td class="text-start"><?= esc($rr['name_ar']) ?><div class="muted small"><?= esc($rr['name_en']) ?></div></td>
                    <td><span class="pill"><?= esc($rr['BranchName']) ?></span></td>
                    <td><?= (int)$rr['quantity'] ?></td>
                    <td><?= (int)$rr['min_quantity'] ?></td>
                    <td><strong class="text-danger"><?= max(0,(int)$rr['min_quantity']-(int)$rr['quantity']) ?></strong></td>
                  </tr>
                <?php else: ?>
                  <tr>
                    <td><?= (int)$rr['variant_id'] ?></td>
                    <td><?= esc($rr['sku']) ?></td>
                    <td class="text-start"><?= esc($rr['name_ar']) ?><div class="muted small"><?= esc($rr['name_en']) ?></div></td>
                    <td><?= (int)$rr['qty'] ?></td>
                    <td><?= (int)$rr['min_qty'] ?></td>
                    <td><strong class="text-danger"><?= max(0,(int)$rr['min_qty']-(int)$rr['qty']) ?></strong></td>
                  </tr>
                <?php endif; ?>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      <?php endif; ?>
    </div>
  </div>

</div>

<!-- ===== Modals ===== -->

<!-- تفضيلاتي -->
<div class="modal fade" id="prefsModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-sliders me-2"></i> تفضيلاتي</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="save_prefs">
      <div class="mb-2">
        <label class="form-label">الفرع الافتراضي</label>
        <select class="form-select" name="default_branch">
          <option value="">— لا شيء —</option>
          <?php foreach ($branches as $bid=>$bn): ?>
            <option value="<?= (int)$bid ?>" <?= (string)$pref['default_branch']===(string)$bid?'selected':''; ?>><?= esc($bn) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="mb-2">
        <label class="form-label">وضع العرض الافتراضي</label>
        <select class="form-select" name="default_mode">
          <option value="by_branch" <?= $pref['default_mode']==='by_branch'?'selected':''; ?>>حسب الفرع</option>
          <option value="by_variant" <?= $pref['default_mode']==='by_variant'?'selected':''; ?>>تجميعي</option>
        </select>
      </div>
      <div class="form-check mb-2">
        <input class="form-check-input" type="checkbox" id="pf_low" name="only_low" <?= $pref['only_low']?'checked':''; ?>>
        <label class="form-check-label" for="pf_low">إظهار المنخفض فقط كمبدئي</label>
      </div>
      <div class="mb-2">
        <label class="form-label">عدد الصفوف</label>
        <input type="number" class="form-control" name="per_page" value="<?= (int)$pref['per_page'] ?>" min="10" max="200">
      </div>
      <div class="text-center"><button class="btn btn-primary px-4">حفظ</button></div>
    </form>
  </div></div>
</div>

<!-- حدّ أدنى -->
<div class="modal fade" id="minModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-gauge-high me-2"></i> تحديث الحد الأدنى</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="update_min">
      <div class="mb-2">
        <label class="form-label">المتغيّر</label>
        <input class="form-control" list="variantsList" id="min_variant_input" placeholder="ID | الاسم | SKU" onblur="parseVariant(this,'min_variant_id')">
        <input type="hidden" name="variant_id" id="min_variant_id">
      </div>
      <div class="mb-2">
        <label class="form-label">الفرع</label>
        <select class="form-select" name="branch_id" id="min_branch_id" required>
          <option value="">— اختر —</option>
          <?php foreach ($branches as $bid=>$bn): ?>
            <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="mb-2">
        <label class="form-label">الحد الأدنى</label>
        <input type="number" class="form-control" name="min_qty" id="min_qty" value="0" min="0">
      </div>
      <div class="text-center"><button class="btn btn-primary px-4">حفظ</button></div>
    </form>
  </div></div>
</div>

<!-- تحويل -->
<div class="modal fade" id="transferModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-right-left me-2"></i> تحويل بين الفروع</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="transfer">
      <div class="mb-2">
        <label class="form-label">المتغيّر</label>
        <input class="form-control" list="variantsList" id="tr_variant_input" placeholder="ID | الاسم | SKU" onblur="parseVariant(this,'tr_variant_id')">
        <input type="hidden" name="variant_id" id="tr_variant_id">
      </div>
      <div class="row g-2">
        <div class="col">
          <label class="form-label">من فرع</label>
          <select class="form-select" name="from_branch" id="tr_from">
            <option value="">— اختر —</option>
            <?php foreach ($branches as $bid=>$bn): ?>
              <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col">
          <label class="form-label">إلى فرع</label>
          <select class="form-select" name="to_branch" id="tr_to">
            <option value="">— اختر —</option>
            <?php foreach ($branches as $bid=>$bn): ?>
              <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
      </div>
      <div class="mb-2">
        <label class="form-label">الكمية</label>
        <input type="number" class="form-control" name="qty" id="tr_qty" value="1" min="1">
      </div>
      <div class="text-center"><button class="btn btn-primary px-4">تنفيذ</button></div>
    </form>
  </div></div>
</div>

<!-- تسوية/ضبط -->
<div class="modal fade" id="adjustModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-scale-balanced me-2"></i> تسوية مخزون</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="adjust">
      <div class="mb-2">
        <label class="form-label">المتغيّر</label>
        <input class="form-control" list="variantsList" id="ad_variant_input" placeholder="ID | الاسم | SKU" onblur="parseVariant(this,'ad_variant_id')">
        <input type="hidden" name="variant_id" id="ad_variant_id">
      </div>
      <div class="mb-2">
        <label class="form-label">الفرع</label>
        <select class="form-select" name="branch_id" id="ad_branch_id" required>
          <option value="">— اختر —</option>
          <?php foreach ($branches as $bid=>$bn): ?>
            <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="mb-2">
        <label class="form-label">فرق الكمية (+ أو −)</label>
        <input type="number" class="form-control" name="delta" id="ad_delta" value="0" step="1">
      </div>
      <div class="mb-2">
        <label class="form-label">السبب</label>
        <select class="form-select" name="reason" id="ad_reason">
          <option value="stock_count">جرد/مطابقة</option>
          <option value="damage">تالف</option>
          <option value="expiry">انتهاء صلاحية</option>
          <option value="correction">تصحيح</option>
          <option value="manual_adjust" selected>يدوي</option>
        </select>
      </div>
      <div class="text-center"><button class="btn btn-primary px-4">تنفيذ</button></div>
    </form>
  </div></div>
</div>

<!-- سجل الحركات -->
<div class="modal fade" id="movModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-clock-rotate-left me-2"></i> سجل الحركات</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <div class="modal-body">
      <div id="movWrap" class="table-responsive">
        <div class="text-muted">جارٍ التحميل…</div>
      </div>
    </div>
  </div></div>
</div>

<datalist id="variantsList">
  <?php foreach ($variantsList as $v): ?>
    <option value="<?= (int)$v['id'] ?> | <?= esc($v['label']) ?>"></option>
  <?php endforeach; ?>
</datalist>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark mode
  const body=document.body, darkSwitch=document.getElementById('darkSwitch');
  if (localStorage.getItem('darkMode')==='true'){ body.classList.add('dark'); darkSwitch.textContent='☀️'; }
  darkSwitch?.addEventListener('click', ()=>{ body.classList.toggle('dark'); const d=body.classList.contains('dark'); localStorage.setItem('darkMode', d); darkSwitch.textContent=d?'☀️':'🌙'; });

  // Helpers
  function parseVariant(inp, hiddenId){
    const hidden=document.getElementById(hiddenId);
    const val=(inp.value||'').trim();
    const id=parseInt(val.split('|')[0],10);
    if (!isNaN(id)) hidden.value=id;
  }

  // Prefill modals from row buttons
  function prefillMin(vid, bid, minq){
    document.getElementById('min_variant_id').value = vid||'';
    document.getElementById('min_variant_input').value = vid? (vid+' | ...'): '';
    document.getElementById('min_branch_id').value = bid||'';
    document.getElementById('min_qty').value = (minq||0);
  }
  function prefillTransfer(vid, fromBid){
    document.getElementById('tr_variant_id').value = vid||'';
    document.getElementById('tr_variant_input').value = vid? (vid+' | ...'): '';
    document.getElementById('tr_from').value = fromBid||'';
  }
  function prefillAdjust(vid, bid){
    document.getElementById('ad_variant_id').value = vid||'';
    document.getElementById('ad_variant_input').value = vid? (vid+' | ...'): '';
    document.getElementById('ad_branch_id').value = bid||'';
    document.getElementById('ad_delta').value = 0;
  }

  // Movements modal
  const movModal = new bootstrap.Modal(document.getElementById('movModal'));
  async function openMov(variant_id, branch_id){
    document.getElementById('movWrap').innerHTML = '<div class="text-muted">جارٍ التحميل…</div>';
    movModal.show();
    try{
      const url = new URL(location.href);
      url.searchParams.set('movements','1');
      url.searchParams.set('variant_id', variant_id);
      if (branch_id) url.searchParams.set('branch_id', branch_id); else url.searchParams.delete('branch_id');
      const res = await fetch(url.toString(), {headers:{'Accept':'application/json'}});
      const data = await res.json();
      if (!data.ok) throw new Error('failed');
      if (!data.rows.length){
        document.getElementById('movWrap').innerHTML = '<div class="text-muted">لا يوجد سجل حركات لهذا العنصر.</div>';
        return;
      }
      let html = `<table class="table table-sm text-center align-middle">
        <thead class="table-light"><tr>
          <th>#</th><th>التاريخ</th><th>الفرع</th><th>التغير</th><th>تكلفة</th><th>المرجع</th><th>ملاحظات</th>
        </tr></thead><tbody>`;
      html += data.rows.map(r=>{
        return `<tr>
          <td>${r.id}</td>
          <td>${(r.created_at||'').toString().replace('T',' ').slice(0,19)}</td>
          <td>${r.branch_id||'-'}</td>
          <td>${r.qty_change}</td>
          <td>${(+r.unit_cost||0).toFixed(2)}</td>
          <td>${(r.ref_type||'-')}${r.ref_id?(' #'+r.ref_id):''}</td>
          <td>${(r.notes||'')}</td>
        </tr>`;
      }).join('');
      html += `</tbody></table>`;
      document.getElementById('movWrap').innerHTML = html;
    } catch(e){
      document.getElementById('movWrap').innerHTML = '<div class="text-danger">تعذر تحميل السجل.</div>';
    }
  }
</script>
</body>
</html>
